/*
**	file:		idt.h
**
**	purpose:	This file contains the format of DOS' internal
**			data table which may be located via the un-
**			documented function call 52h.
**
**			This information has been gathered from many
**			sources, including public bulletin boards,
**			magazine articles, books, and personal
**			exploration.  One significant source of the
**			structure of this mess is Jim Kyle. Through
**			his program CVTTRC, downloaded from Compuserve,
**			many holes were filled.
**
**	author:		Bill Parrott
**
**			Copyright (c) 1988, Bill Parrott
**			  All Rights Reserved
*/

#pragma	pack(1)

/*
**	type definitions
*/

typedef	unsigned char	byte;
typedef	unsigned int	word;
typedef	unsigned long	dword;
typedef	unsigned int	segment;

typedef	struct idt	IDT;
typedef	struct devhdr	DEVHDR;
typedef	struct pdpt	PDPT;
typedef	struct ldt	LDT;
typedef	struct mdb	MDB;
typedef	struct bcb	BCB;
typedef	struct chain	CHAIN;
typedef	struct ifcb	IFCB;
typedef	struct time	TIME;
typedef	struct date	DATE;
typedef	struct psp	PSP;



/*
**	This structure defines a device driver header.  Device drivers
**	are linked in a chain by MS-DOS during system boot up.  The
**	sequence of this linkage is:
**
**		NUL
**		<installable device drivers>
**		CON
**		AUX
**		PRN
**		CLOCK$
**
**	Installable device drivers are loaded and appear in the
**	sequence encountered in CONFIG.SYS.  The NUL device
**	header is listed as part of the internal data table
**	(below) and may be used as a starting point in the list.
*/

struct devhdr
	{
	DEVHDR	far *nextheader;	/* address of next header	*/
	word	attributes;		/* device attributes		*/
	word	strategy;		/* strategy entry point		*/
	word	intrupt;		/* interrupt entry point	*/
	byte	devname[8];		/* device name			*/
	};




/*
**	This is the definition of the internal data table.  Its
**	address may be found by subtracting 8 from the value
**	returned in (bx) from DOS function 52h and combining
**	that offset with the segment returned in (es).
**
**	This table format is not correct for versions of DOS
**	prior to 3.0.
*/

struct idt
	{
	BCB	far *curbcb;		/* address of current buffer		*/
	MDB	far *arena;		/* start of memory arena		*/
	PDPT	far *pdpt;		/* physical drive parameter table	*/
	CHAIN	far *ifcb;		/* internal file control blocks		*/
	void	far *clockdev;		/* address of clock device driver	*/
	void	far *condev;		/* address of console device driver	*/
	word	max_ssize;		/* maximum allowed sector size		*/
	BCB	far *bcbhead;		/* start of buffer list			*/
	LDT	far *ldt;		/* address of logical device table	*/
	CHAIN	far *fcbt;		/* file control block table		*/
	word	nprotfcbs;		/* number of protected fcbs (FCBS=?,n)	*/
	byte	nphysdrives;		/* number of physical devices		*/
	byte	nlogdrives;		/* number of logical devices (in ldt)	*/
	DEVHDR	nuldev;			/* NUL device header			*/
	};




/*
**	This structure defines the physical device table.  An
**	individual entry in the table may be obtained by calling
**	DOS interrupt 21H with function 32H, (dl) set to the
**	drive desired (0=default, 1=A, 2=B, ...).  Returned
**	value in (al) = FFH if the drive does not exist, else
**	00H, ds:bx points to the desired entry.
**
**	The head of the physical device table may be found in
**	the internal data table, described above.
*/

struct pdpt
	{
	byte	drive;			/* Drive: 0=A, 1=B, etc.	*/
	byte	dunit;			/* Unit within driver (0,1,...)	*/
	word	sectorsize;		/* bytes per sector		*/
	byte	clustersize;		/* sectors per cluster - 1	*/
	byte	shift;			/* cluster to sector shift	*/
	word	bootsectors;		/* number of reserved sectors	*/
	byte	fats;			/* number of file alloc. tables	*/
	word	rootsize;		/* number of root dir. entries	*/
	word	dataoffset;		/* sector # of cluster 2 (data)	*/
	word	lastcluster;		/* number of clusters + 1	*/
	byte	fatsize;		/* sectors for FAT		*/
	word	diroffset;		/* sector number of directory	*/
	DEVHDR	far *devheader;		/* address of device header	*/
	byte	mediabyte;		/* media descriptor byte	*/
	byte	accessflag;		/* 0 if disk has been accessed	*/
	PDPT	far *nextentry;		/* address of next entry	*/
	};


/*
**	This structure defines the logical device table.  This
**	table is utilized by MS-DOS to keep track of logical
**	drives.  The number of entries in the table is kept
**	in the internal device table (above) in nlogdrives.
**	The number of logical drives is set to the higher
**	of LASTDRIVE= in CONFIG.SYS or the number of physical
**	drives.
**
**	Some general observations (not necessarily rules!!):
**
**	code == 00 if logical device is invalid
**	code == 40 if logical device directly associated with physical device
**	code == 50 if logical device is SUBST'ed
**
**	flag == 001b if drive is SUBST'ed
*/

struct ldt
	{
	char	currentdir[68];		/* current directory for this drive	*/
	byte	code;			/* ??  00, 40 or 50			*/
	PDPT	far *pdpt;		/* physical device for this logical dev	*/
	word	curdir;			/* directory sector			*/
	word	fubar[2];		/* ??  FFFFh FFFFh			*/
	word	flag;			/* ??  0002, 0008, 001b			*/
	};



/*
**	This structure defines a memory descriptor.  These are used
**	by MS-DOS to keep track of allocated memory.  The address
**	of the first descriptor is found in the idt (above).
*/

struct mdb
	{
	byte	type;			/* 'M' if descriptor, 'Z' if last one	*/
	segment	owner;			/* PSP of owner				*/
	word	length;			/* length of this block (paragraphs)	*/
	};



/*
**	This structure defines the buffer control block. The
**	information here is incomplete because it comes entirely
**	from CVTTRC.C.
*/

struct bcb
	{
	BCB	far *nextbcb;		/* address of next BCB			*/
	byte	logdrive;		/* logical drive associated with buffer	*/
	byte	action;			/* action code				*/
	word	logsec;			/* logical sectoe #			*/
	byte	nfats;			/* # of FATs				*/
	byte	sectorsperfat;		/* sectors per FAT			*/
	PDPT	far *pdpt;		/* pdpt associated with buffer		*/
	word	fubar;			/* ?? (probably filler to align buffer)	*/
	byte	buffer[512];		/* the buffer				*/
	};


/*
**	This structure is used to define the length of a chain of
**	internal file control blocks.  By itself, it really does
**	nothing, however it provides the mechanism for locating
**	IFCBs.
*/

struct chain
	{
	CHAIN	far *nextchain;		/* link to next chain			*/
	word	nentries;		/* number of entries in this link	*/
	};



/*
**	This structure defines the internal file control block
**	used by MS-DOS to track open files.  The total number of
**	these in the internal file control block chain is determined
**	by FILES= in CONFIG.SYS.  The total number of these in the
**	FCB table chain is determined by FCBS= in CONFIG.SYS.
*/

struct date
	{
	unsigned day	: 5;		/* day (1-31)				*/
	unsigned month	: 4;		/* month (1-12)				*/
	unsigned year	: 7;		/* 1980 - year (0-127)			*/
	};


struct time
	{
	unsigned second	: 5;		/* seconds / 2 (0-29)			*/
	unsigned minute	: 6;		/* minute (0-59)			*/
	unsigned hour	: 5;		/* hour (0-23)				*/
	};


struct ifcb
	{
	word	nopens;			/* # opens active			*/
	word	mode;			/* open mode, 0=read, 1=write, 2=r/w	*/
	byte	fileattr;		/* file attribute (flags) byte		*/
	word	devattr;		/* device attribute word		*/
	PDPT	far *pdpt;		/* pdpt associated with file		*/
	word	firstcluster;		/* 1st cluster of file			*/
	TIME	filetime;		/* file time				*/
	DATE	filedate;		/* file date				*/
	dword	filesize;		/* file size (bytes)			*/
	dword	filepos;		/* current file position		*/
	word	nclusters;		/* # of clusters in file		*/
	word	curcluster;		/* current cluster			*/
	word	dirsector;		/* directory sector			*/
	byte	dirindex;		/* index in directory sector		*/
	byte	filename[8];		/* file name				*/
	byte	fileext[3];		/* file type (extension)		*/
	word	fubar[3];		/* ???					*/
	segment	owner;			/* PSP of file owner			*/
	word	fubar2;			/* ???					*/
	};



/*
**	This structure define the program segment prefix.  There
**	is a PSP associated with every process executing under
**	MS-DOS.  Note that in this definition, the FCB's are not
**	defined.  Since no one (I HOPE!) uses these any more, and
**	even if they do they generally have to copy them first,
**	this shouldn't really be a big hit.
*/

struct psp
	{
	word	wmboot;			/* program termination (int 20h)	*/
	segment	himem;			/* segment of end-of-memory		*/
	byte	reserved1;		/* reserved				*/
	byte	calldos[5];		/* far call to DOS function dispatcher	*/
	void	(far *termhndlr)();	/* pointer to terminate handler		*/
	void	(far *breakhndlr)();	/* pointer to crtl/break handler	*/
	void	(far *crithndlr)();	/* pointer to critical error handler	*/
	segment	parentpsp;		/* PSP of parent process		*/
	byte	handles[20];		/* handle table (FF=available)		*/
	segment	environment;		/* environment block			*/
	byte	reserved2[4];		/* reserved				*/
	word	nhandles;		/* handle table size			*/
	dword	phandles;		/* handle table address			*/
	byte	reserved3[24];		/* reserved				*/
	word	int21;			/* int 21h DOS call			*/
	byte	farret;			/* retf					*/
	byte	reserved4[9];		/* reserved				*/
	byte	fcb1[16];		/* 1st CP/M-style FCB			*/
	byte	fcb2[16];		/* 2nd CP/M-style FCB			*/
	byte	extra[4];		/* overflow from FCBs			*/
	byte	cmdlen;			/* length of command line parameters	*/
	byte	cmdline[127];		/* command line parameters		*/
	};


