/* TiffDiag.c - Tiff diagnostics routines
 */
#include "aldtypes.h"
#include "imtypes.h"
#include "imag.h"
#include "imtiff.h"
#include "tiff.h"
#include "ImErr.h"
#include "ImErr2.h"		/* application-dependent! for WarningAlert proto and IA_PLACE_IMAGE */

extern char ErrorMessages[];

#define MAX_RECOMMENDED_STRIP	(12*1024L)

/* check the tiff structure for bad values and combinations of values
 */
RC CheckTiff (x)
register IMAG	*x;
{
		register RC		err = SUCCESS;
		
		if (!x->eNewSubfileType) {
			WARN ((HWND)NULL, IM_WARNING, IM_NO_NEWSUBFILETYPE);
		}

		/* ImageWidth,ImageLength
		 */
		if (!x->eImageWidth) {
			DBMSG(("CheckTiff: no width\n"));
			err = IM_NO_WIDTH;
			goto cu0;
		}
		if (!x->eImageLength) {
			DBMSG(("CheckTiff: no length\n"));
			err = IM_NO_LENGTH;
			goto cu0;
		}
		if (x->iImageWidth == 0) {
			DBMSG(("CheckTiff: 0 width\n"));
			err = IM_BAD_WIDTH;
			goto cu0;
		}
		if (x->iImageLength == 0) {
			DBMSG(("CheckTiff: 0 length\n"));
			err = IM_BAD_LENGTH;
			goto cu0;
		}

		if (!x->eStripOffsets) {
			DBMSG(("CheckTiff: no offsets\n"));
			err = IM_NO_OFFSETS;
			goto cu0;
		}
		
		/* check bit depth */
		{
			WORD	Bits;
			
			Bits = x->iBitsPerSample;
			if (Bits == 1 || Bits == 4 || Bits == 6 || Bits == 8) {
				if (Bits == 6) {
					WARN ((HWND)NULL, IM_WARNING, IM_FADING_BITDEPTH);
				}
			} else {
				DBMSG(("CheckTiff: bad bitspersample\n"));
				err = IM_BAD_BPS;
				goto cu0;
			}
		}
		
		/* if Samples > 1, replicate BitsPerSample, to keep everything kosher
		 */
		if (x->iSamples != 1 && x->iSamples != 3) {
			DBMSG(("CheckTiff: bad spp\n"));
			err = IM_BAD_SPP;
			goto cu0;
		}
		if (x->iSamples > 1) {
			WORD	bpsBuf[4];	/* 3 for RGB, 4 for CMYK */
			RC		err2;
			register int		ii;
			
			if (x->tf[X_BITSPERSAMPLE].Tlength != x->iSamples) {
			
				WARN ((HWND)NULL, IM_WARNING, IM_BAD_NUM_BITS);
			}
			
			if (x->iPhotometricInterpretation != TIFFRGB) {
				WARN ((HWND)NULL, IM_WARNING, IM_COLOR_CLASH);
			}
		}
		
		/* planarconfiguration
		 */
		if (x->ePlanar && x->iSamples > 1 && x->iPlanar != CHUNKY) {
			DBMSG(("CheckTiff: bad planar\n"));
			err = IM_BAD_PLANAR;
			goto cu0;
		}
		
		/* photometric */
		{
			WORD	Photo;
			
			Photo = x->iPhotometricInterpretation;
			if (Photo != WHITEZERO && Photo != BLACKZERO && Photo != TIFFRGB) {
				DBMSG(("CheckTiff: unsupported photometricinterp = %u\n", Photo));
				err = IM_BAD_PHOTO;
				goto cu0;
			}
			if (!x->ePhotometricInterpretation) {
				WARN ((HWND)NULL, IM_WARNING, IM_NO_PHOTO);
			}
		}
		
		/* compression
		 */
		{
			WORD	Compr;
			
			Compr = x->iCompression;
			
			if (Compr == PACKINTOBYTES) {
				WARN ((HWND)NULL, IM_WARNING, IM_NO_COMPR);		/* no compression */
			} else if (Compr == CCITT1D || Compr == COMPR5 || Compr == TIFFPACKBITS) {
				/* ok */
			} else {
				WARN ((HWND)NULL, IM_WARNING, IM_BAD_COMPR);	/* unknown compression */
			}
			
			if (Compr == TIFFPACKBITS && x->iBitsPerSample != 1) {
				WARN ((HWND)NULL, IM_WARNING, IM_PB_BITSNOTONE);	/* unknown compression */
			}
		}
		
		/* Predictor */
		{
			WORD	Pred;
			
			Pred = x->iPredictor;
			if (Pred == PREDICTOR_NONE) {
			} else if (Pred == PREDICTOR_HDIFF) {
				if (x->iBitsPerSample != 8) {
					DBMSG(("CheckTiff: horiz diff but bits=%u\n", x->iBitsPerSample));
					err = IM_PRED_MISMATCH;
					goto cu0;
				}
			} else {
				DBMSG(("CheckTiff: unknown predictor\n"));
				err = IM_BAD_PREDICT;
				goto cu0;
			}
		}
		
		/* make RowsPerStrip no larger than ImageLength, for future convenience
		 */
		if (x->iRowsPerStrip > x->iImageLength)
			x->iRowsPerStrip = x->iImageLength;
		
		/* check strip size
		 *
		 * TODO: do we want to store dwStripBytes around for future convenience?
		 * If so, make sure all the other imports do the same ... or do it in fr.c?
		 */
		{
			DWORD	dwBytesPerRow, dwStripBytes;
			
			dwBytesPerRow = UBPR(x->iImageWidth, x->iBitsPerSample, x->iSamples);
			dwStripBytes = (DWORD)x->iRowsPerStrip * dwBytesPerRow;
			
			if (dwStripBytes > MAX_RECOMMENDED_STRIP) {
				WARN ((HWND)NULL, IM_WARNING, IM_LARGE_STRIP);
			}		
		}
		
		/* complain if there are no StripByteCounts
		 */
		if (!x->eStripByteCounts) {
			WARN ((HWND)NULL, IM_WARNING, IM_NO_BYTECOUNTS);
		}
		
		/* fatal error if advertised number of offsets or counts does not
		 * equal the computed number
		 */
		{
			WORD	NumStrips;
			
			if (x->iRowsPerStrip == 0) {
				DBMSG(("CheckTiff: rps = 0\n"));
				err = IM_BAD_ROWSPERSTRIP;
				goto cu0;
			}
			NumStrips = (x->iImageLength + x->iRowsPerStrip - 1) / x->iRowsPerStrip;
			
			if (x->eStripOffsets) {
				if (NumStrips != (WORD)x->tf[X_STRIPOFFSETS].Tlength) {
					DBMSG(("CheckTiff: bad num offsets\n"));
					err = IM_BAD_NUM_OFF;
					goto cu0;
				}
			}
			if (x->eStripByteCounts) {
				if (NumStrips != (WORD)x->tf[X_STRIPBYTECOUNTS].Tlength) {
					DBMSG(("CheckTiff: bad num counts\n"));
					err = IM_BAD_NUM_COUNTS;
					goto cu0;
				}
			}
		}
		
		/* return
		 */
cu0:	return err;

} /* end of CheckTiff */
