/* ----------------------------------------------------------------------------	*/
/*	Originators:																*/
/*  	Tom Arnold,	Quark, Inc.,	300 S. Jackson, 	Denver, 	CO 80209	*/
/*		Ron Perry, 	Atex, Inc.,		165 Lexington Road,	Billerica, 	MA 01821	*/
/* ----------------------------------------------------------------------------	*/
/*	The source code in this file is provided by Quark and Atex for use in the	*/
/*	development of a Publishing Interchange Language. The code provided herein	*/
/*	is non-proprietary, non-copyrighted, and is for use in the public domain.	*/
/*	The source code contained herein is provided "as is".						*/
/* ----------------------------------------------------------------------------	*/
/*	QUARK AND ATEX DISCLAIM ALL WARRANTIES, EXPRESSED OR IMPLIED, WITH REGARD	*/ 
/*	TO THE ENCLOSED SOURCE CODE. QUARK AND ATEX DISCLAIM ANY IMPLIED WARRANTY,	*/
/*	INCLUDING, BUT NOT LIMITED TO ANY IMPLIED WARRANTY OF FITNESS FOR A			*/
/*	PARTICULAR PURPOSE OR MERCHANTABILITY.										*/
/* ----------------------------------------------------------------------------	*/

/*$au ------------------------------ audits -----------------------------------
    Author: Tom Arnold
            Quark, Inc.
            300 South Jackson
            Denver, Colorado 80209
            (303) 934-2211

    Ver     Date       Who  Etc
    v01.02  25-apr-91  rnp	Update for PIL 5.0
    v01.01  10-apr-91  tla	Added audit trail, changed cnt_buffering_size
---------------------------------------------------------------------------- */


/* ------------- */
/* Include Files */
/* ------------- */


#include	<stdio.h>	/* For fopen, fclose, etc.					*/
#include	<stdlib.h>	/* malloc, realloc							*/
#include	"pildefs.h"	/* Pick up PIL data structures and defines	*/


/* --------------------- */
/* Defines for this file */
/* --------------------- */


#define		DEFAULT_INPUT_FILE		"in.pil"
#define		DEFAULT_OUTPUT_FILE		"out.pil"
#define 	WBUFSIZE 				512
#define 	RBUFSIZE 				512
#define		PIL_BUFSIZE_INCREASE 	2048


/* ----------- */
/* Static data */
/* ----------- */


static char *input = NULL;						/* input file name 			 */
static char *output = NULL;						/* output file name 		 */
static FILE *infile = NULL;						/* file number for reading   */
static FILE *outfile = NULL;					/* file number for writing   */
static char writebuf[WBUFSIZE];					/* buffer for writing        */
static int wbufpos = 0;							/* initial position in buf   */
static char readbuf[RBUFSIZE];					/* buffer for reading		 */
static int rbufpos = 0;							/* initial postion in buf	 */
static int remaining = 0;						/* bytes in readbuf			 */
static char *pilbuf = NULL;						/* pointer to buffer for api */
static size_t pil_buffer_size = 1024;			/* default size for pilbuf 	 */
static size_t cnt_buffering_size = 512;			/* def sz of content blocks  */
static char *asi_names[] = 						/* Application Spec. Items	 */
{												/* -----------------------   */ 
	"CompanyName-time-stamp",					/* Time Stamp attribute		 */
	"CompanyName-editor-name",					/* Editor Name attribute	 */
	"CompanyName-text-format-name"				/* Text Format attribute	 */
};												/* End of Appl Spec Items	 */
static int nlayouts = 0;						/* total layouts seen 		 */
static int curobject = 0;						/* # objects on cur. layout  */
static int nobjects = 0;						/* total objects seen 		 */
static PIL_COMP_TYPE pil_pig;					/* type of biggest object    */
static PIL_UINT32 largest;						/* # bytes it used           */


/* ---------- */
/* Prototypes */
/* ---------- */


#ifdef 		PIL_ANSI_PROTOTYPES
int			ap_init(char *, char *,size_t);
int			ap_getc(PIL_VOID);
PIL_VOID	ap_putc(PIL_UINT8);
PIL_VOID	error_report(PIL_ERROR);
int			flush_writebuf(PIL_VOID);
PIL_VOID 	ap_cleanup(PIL_VOID);
PIL_VOID 	printstats(PIL_VOID);
PIL_VOID 	getstats(pil_component *);
PIL_VOID 	clearstats(PIL_VOID);
int 		restart_parser(PIL_VOID);
#else
int			ap_init();
int			ap_getc();
PIL_VOID	ap_putc();
PIL_VOID	error_report();
int			flush_writebuf();
PIL_VOID 	ap_cleanup();
PIL_VOID 	printstats();
PIL_VOID 	getstats();
PIL_VOID 	clearstats();
int 		restart_parser();
#endif


/* ----------------------------------------------------------------------------	*\
  ap_init

	Initialize the application: 
		1. allocate memory for the PIL buffer
		2. open the necessary files and prepare for parsing and/or generator

	Entry:	input file, output file, and PIL buffer size

	Exit:	0 upon success, -1 if failure
\* ----------------------------------------------------------------------------	*/

int ap_init (in, out, bufsize)
char *in;
char *out;
size_t bufsize;
{
	/* --------------------------------- */
	/* initialize read and write buffers */
	/* --------------------------------- */
	wbufpos = rbufpos = remaining = 0;


	/* ------------------- */
	/* allocate PIL buffer */
	/* ------------------- */
	if ((pilbuf = (char *)malloc(bufsize)) == NULL) return(-1);


	/* -------------------------- */
	/* open input file, if needed */
	/* -------------------------- */
	if (*in) 
	{
		if ((infile = fopen (in, "r")) == NULL) return (-1);
	}
	

	/* --------------------------- */
	/* open output file, if needed */
	/* --------------------------- */
	if (*out) 
	{
		if ((outfile = fopen (out, "w")) == NULL) return (-1);
	}
	return (0);
}


/* ----------------------------------------------------------------------------	*\
  ap_cleanup

	Clean up the application:
		1. Close all files
		2. deallocate PIL buffer

	Entry:	none

	Exit:	none
\* ----------------------------------------------------------------------------	*/

PIL_VOID ap_cleanup()
{
	if (infile != NULL) 
	{
		fclose (infile);
		infile = NULL;
	}
	
	if (outfile != NULL) 
	{
		flush_writebuf();
		wbufpos = 0;
		fclose (outfile);
		outfile = NULL;
	}
	
	if (pilbuf) 
	{
		free (pilbuf);
		pilbuf = NULL;
	}
}


/* ----------------------------------------------------------------------------	*\
  ap_getc

	The applications "getchar" function. A pointer to this function is passed to 
	the api via pil_pg_init(). This routine knows what file to read from, and 
	checks to see that it has been open. If the file number is -1, it indicates 
	that either the file is not open, or an error has occurred on a previous read.
	In either case, ap_getc just returns.
 
	Entry:	None.

	Exit:	-1 if error or eof, else character read, cast as an int.
\* ----------------------------------------------------------------------------	*/

int ap_getc()
{
	if (infile == NULL) return (-1);
	if (remaining == 0) 
	{
		if ((remaining = (int) fread (readbuf, (size_t) 1, 
			(size_t) RBUFSIZE, infile)) == 0) 
		{
			infile = NULL;
			return (-1);
		}
		rbufpos = 0;
	}
	remaining--;
	return ((int)readbuf[rbufpos++]);
}


/* ----------------------------------------------------------------------------	*\
  ap_putc

    The applications "putchar" function. A pointer to this function is 
	passed to the api via pil_pg_init(). This routine knows what file to 
	write to, and checks to see that it has been opened. If the file 
	number is -1, it indicates that either the file is not open, or 
	an error has occurred on a previous write. Otherwise, ap_putc 
	buffers up characters and writes them when the buffer is full.

    NOTE: to ensure that all characters are written, flush_writebuf()
    must be called after the last call to ap_putc.

    Entry:  PIL_UINT8 to write.

    Exit:   None
\* ----------------------------------------------------------------------------	*/

#ifdef 		PIL_ANSI_PROTOTYPES
PIL_VOID ap_putc(PIL_UINT8 c)
#else
PIL_VOID ap_putc(c)
PIL_UINT8 c;
#endif
{
    if (outfile == NULL) return ;
    if (wbufpos == WBUFSIZE) 
	{
        if (flush_writebuf() != wbufpos) outfile = NULL;
        wbufpos = 0;
    }
    writebuf[wbufpos++] = c;
}


/* ----------------------------------------------------------------------------	*\
  flush_writebuf

    Flush the write buffer.

    NOTE: to ensure that all characters are written, flush_writebuf()
    must be called after the last call to ap_putc.

    Entry:  PIL_UINT8 to write.

    Exit:   number of bytes written to the file
\* ----------------------------------------------------------------------------	*/

int flush_writebuf ()
{
    if (outfile == NULL) return (0);
    return ((int) fwrite (writebuf, (size_t) 1, (size_t) wbufpos, outfile));
}


/* ----------------------------------------------------------------------------	*\
  error_report

	The applications error reporting function.
 
	Entry:	PIL_ERROR code.

	Exit:	none
\* ----------------------------------------------------------------------------	*/

PIL_VOID error_report(err)
PIL_ERROR err;
{
	switch (err) 
	{
		case PIL_OUT_OF_MEMORY:
			printf("PIL_ERROR: Not enough room left in pil buffer\n");
			break;
		case PIL_END_OF_FILE:
			printf("PIL_ERROR: End of file was encountered on a read\n");
			break;
		case PIL_NO_BUFPTR:
			printf("PIL_ERROR: The pil buffer was not provided\n");
			break;
		case PIL_NO_ASALISTPTR:
			printf("PIL_ERROR: Asa list was expected, but not provided\n");
			break;
		case PIL_BAD_STRUCT:
			printf("PIL_ERROR: Bad structure found when generating\n");
			break;
		case PIL_BAD_SYNTAX:
			printf("PIL_ERROR: Parser encountered illegal input, line %lu\n",
					pil_get_line_num());
			break;
		case PIL_BAD_FILE_TYPE:
			printf("PIL_ERROR: Input to the parser is NOT a PIL file\n");
			break;
		case PIL_PREV_ERR:
			printf("PIL_ERROR: Previous error encountered by parser\n");
			break;
		case PIL_BAD_CBUF_SIZE:
			printf("PIL_ERROR: Invalid (zero) content buffering size\n");
			break;
		case PIL_BAD_CMPNT_TYPE:
			printf("PIL_ERROR: Invalid component type\n");
			break;
		case PIL_BAD_VALUE_TYPE:
			printf("PIL_ERROR: Invalid pil_value type\n");
			break;
		case PIL_NO_RW_FUNCS:
			printf("PIL_ERROR: Both getc and putc func ptrs are null\n");
			break;
		case PIL_NO_R_FUNC:
			printf("PIL_ERROR: getc function pointer is null\n");
			break;
		case PIL_NO_W_FUNC:
			printf("PIL_ERROR: putc function pointer is null\n");
			break;
		case PIL_UNKN_HDR_BORD:
			printf("PIL_ERROR: Unknown byte order in cnt data hdr\n");
			break;
		case PIL_BAD_UNITS:
			printf("PIL_ERROR: Invalid Units (canvas or object)\n");
			break;
		case PIL_ZERO_DIMENSION:
			printf("PIL_ERROR: Object or Canvas with zero dimension\n");
			break;
		case PIL_BAD_NTE:
			printf("PIL_ERROR: A name table entry is missing a field\n");
			break;
		case PIL_OPEN_CLIPPER:
			printf("PIL_ERROR: An open shape can't be used as a clipper\n");
			break;
		case PIL_ZERO_SHAPE:
			printf("PIL_ERROR: A shape can't have zero dimension\n");
			break;
		case PIL_UNKN_SHAPE:
			printf("PIL_ERROR: Unknown shape\n");
			break;
		case PIL_UNKN_PATHPT:
			printf("PIL_ERROR: Unknown path point type\n");
			break;
		case PIL_ZERO_LENGTH:
			printf("PIL_ERROR: A line can't have zero length\n");
			break;
		case PIL_NEED_MOVE_TO:
			printf("PIL_ERROR: Path must start with a move-to\n");
			break;
		case PIL_OPEN_CONTAINER:
			printf("PIL_ERROR: A container must be a closed shape\n");
			break;
		case PIL_UNKN_CM:
			printf("PIL_ERROR: An unknown color model\n");
			break;
		case PIL_UNKN_RENDEROP:
			printf("PIL_ERROR: An unknown render operation\n");
			break;
		case PIL_UNKN_FILLRULE:
			printf("PIL_ERROR: An unknown fill rule\n");
			break;
		case PIL_UNKN_LINEPOS:
			printf("PIL_ERROR: Invalid Line position\n");
			break;
		case PIL_BAD_OBJID:
			printf("PIL_ERROR: object id list with null id pointer\n");
			break;
		case PIL_BAD_MITERLIM:
			printf("PIL_ERROR: Miter limit must be >= 1.0\n");
			break;
		case PIL_BAD_COLOR:
			printf("PIL_ERROR: Color value out of range\n");
			break;
		case PIL_BAD_HEADER:
			printf("PIL_ERROR: Incorrect values in a cnt header\n");
			break;
		case PIL_BAD_ARGS:
			printf("PIL_ERROR: Invalid arguments passed to a pil func\n");
			break;
		case PIL_WRONG_CMPNT:
			printf("PIL_ERROR: Incorrect component type provided\n");
			break;
		case PIL_UNKN_HDR_ENC:
			printf("PIL_ERROR: Unknown data encoding in a content header.\n");
			printf("Encoding must be ascii or binary\n");
			break;
		case PIL_OK:
		default:
			break;
	}
}


/* ----------------------------------------------------------------------------	*\
  restart_parser

	Restart the parser. This routine is invoked when the current PIL buffer 
	is not big enough.
 
	Entry:	none	

	Exit:	0 if everything is OK, -1 if a failure occurs	
\* ----------------------------------------------------------------------------	*/

int restart_parser()
{
	PIL_ERROR ret_code;
	ap_cleanup();
	pil_buffer_size += PIL_BUFSIZE_INCREASE;
	if (ap_init (input,"",pil_buffer_size)) 
	{
		printf("Unable to reinitialize parser.\n");
		return (-1);
	}

	ret_code = pil_pg_init
	(
		pilbuf,
		(PIL_UINT32) pil_buffer_size,
		ap_getc,
		ap_putc,
		(PIL_UINT32) cnt_buffering_size,
		(PIL_INT32) 3,
		asi_names
	);

	if (ret_code != PIL_OK) return (-1);
	return (0);
}


/* ----------------------------------------------------------------------------	*\
  clearstats
	
	Clear the application statistics. These are being maintained as an
	example of simple application processing.

	Entry:	none	

	Exit:	none	
\* ----------------------------------------------------------------------------	*/

PIL_VOID clearstats()
{
	nlayouts = 0;					/* total layouts seen 		 */
	curobject = 0;					/* # objects on cur. layout  */
	nobjects = 0;					/* total objects seen 		 */
	pil_pig = 0;					/* type of biggest object    */
	largest = 0;					/* # bytes it used           */
}


/* ----------------------------------------------------------------------------	*\
  getstats
	
	Update application statistics and print out various PIL component data
	elements.

	Entry:	The PIL component encountered by the parser.

	Exit:	none	
\* ----------------------------------------------------------------------------	*/

PIL_VOID getstats(pc)
pil_component *pc;
{
	PIL_UINT32 remaining;

	/* ------------------------------------------------------------------- */
	/* keep a running count of greatest memory used and the component type */ 
	/* ------------------------------------------------------------------- */
	if (pil_get_mem_avail(&remaining) != PIL_OK) 
	{
		error_report(pil_last_error());
		return;
	}


	/* --------------------------------------------- */
	/* Check to see if this is the largest component */
	/* --------------------------------------------- */
	if (pil_buffer_size - remaining > largest) 
	{
		largest = pil_buffer_size - remaining;
		pil_pig = pc->type;
	}


	/* ------------------------------------------- */
	/* Gather statistics based upon component type */
	/* ------------------------------------------- */
	switch (pc->type) 
	{
		case PIL_LAYOUT_START_C:
		{
			nlayouts++;
			curobject = 0;
			printf("Layout: #%d, Name: %s, Ending Line: #%lu\n", nlayouts, 
			pc->ls.name, pil_get_line_num());
			break;
		}
		case PIL_CANVAS_C:
		{
			printf("\tCanvas, Ending Line #%lu\n", pil_get_line_num());
			if (pc->pc.username) printf("\tCanvas Username: %s\n",pc->pc.username);
			if (pc->pc.cantype) printf("\tCanvas Type: %s\n",pc->pc.cantype);
			break;
		}
		case PIL_OBJECT_C:
		{
			nobjects++;
			curobject++;
			printf("\tObject #%d, Ending Line #%lu\n", curobject, pil_get_line_num());
			if (pc->po.username) printf("\tObject Username: %s\n",pc->po.username);
			switch (pc->po.objtype)
			{
				case PIL_OBJ_UNKNOWN:	break;
				case PIL_OBJ_TEXT:		printf("\tObject Type: Text\n"); 		break;
				case PIL_OBJ_PICTURE:	printf("\tObject Type: Picture\n"); 	break;
				case PIL_OBJ_PRIMITIVE:	printf("\tObject Type: Primitive\n"); 	break;
			}
			if (pc->po.rcname) printf("\tObject RCName: %s\n",pc->po.rcname);
			if (pc->po.asi != NULL) 
			{
				if (pil_strcmp (pc->po.asi->asi_name,"sample-editor-name") == 0) 
				{
					printf("\tObject Editor: %s\n",
					pc->po.asi->asa->value->data.string);
				}
			}
			break;
		}
		default:
		{
			break;
		}
	}
}


/* ----------------------------------------------------------------------------	*\
  printstats
	
	Print out the application statistics. This is invoked when the parser
	has encountered the end of the PIL stream.

	Entry:	none	

	Exit:	none	
\* ----------------------------------------------------------------------------	*/

PIL_VOID printstats()
{
	printf("Finished Parsing %s.\n",input);
	printf("\t%d layouts processed, %d objects processed.\n",nlayouts, nobjects);
	printf("\tLargest pil_component was a ");
	switch(pil_pig) 
	{
		case PIL_LAYOUT_START_C:
			printf("PIL_LAYOUT_START ");
			break;
		case PIL_NAME_TABLE_C:
			printf("PIL_NAME_TABLE ");
			break;
		case PIL_CANVAS_C:
			printf("PIL_CANVAS ");
			break;
		case PIL_OBJECT_C:
			printf("PIL_OBJECT ");
			break;
		case PIL_TEXT_FLOW_C:
			printf("PIL_TEXT_FLOW ");
			break;
		case PIL_GROUP_C:
			printf("PIL_GROUP ");
			break;
		case PIL_LAYOUT_END_C:
			printf("PIL_LAYOUT_END ");
			break;
		case PIL_CONTENT_START_C:
			printf("PIL_CONTENT_START ");
			break;
		case PIL_CONTENT_HDR_C:
			printf("PIL_CONTENT_HDR ");
			break;
		case PIL_CONTENT_DATA_C:
			printf("PIL_CONTENT_DATA ");
			break;
		case PIL_CONTENT_END_C:
			printf("PIL_CONTENT_END ");
			break;
	}
	printf("\n\tthat occupied %lu bytes in the buffer.\n",largest);
}


/* ----------------------------------------------------------------------------	*\
  main
	
	This main demonstrates the use of the PIL ToolKit to parse PIL files.
	It performs the following:

		1. Init's the application (Prepares for file I/O, allocates memory,...)
		2. Init's the PIL ToolKit via pil_pg_init (...)
		3. Loops through the PIL stream and processes each PIL component
		   encountered. The loop is terminated upon encountering the end
		   of the PIL stream or a parser error. Note that if insufficient
		   memory has been allocated for the PIL buffer this main loop
		   will invoke a series of functions to allocate more memory and
		   restart parsing at the beginning of the file.
\* ----------------------------------------------------------------------------	*/

main (argc,argv)
int	argc;
char *argv[];
{
	PIL_ERROR		err = PIL_OK;
	PIL_BOOL		keep_parsing;
	pil_component	*pc;


	/* ------------------------------------------------------------ */
	/* Get command line input file or default to DEFAULT_INPUT_FILE */
	/* ------------------------------------------------------------ */
	if (argc > 1) input = argv[1]; 
	else input = DEFAULT_INPUT_FILE;


	/* ---------------------------------------------------- */
	/* Call our application specific initialization routine	*/
	/* to open files for I/O, and allocate a PIL buffer 	*/
	/* ---------------------------------------------------- */
	if (ap_init (input,"",pil_buffer_size)) 
	{
		printf ("Cannot initialize application\n");
		exit (-1);
	}
	

	/* ---------------------------------------------------- */
	/* If that succeeds, set up the PIL API for business 	*/
	/* ---------------------------------------------------- */
	err = pil_pg_init
    (
        pilbuf,                             /* PIL Buffer address                   */
        (PIL_UINT32) pil_buffer_size,  		/* Size of PIL Buffer                   */
        ap_getc,                          	/* Application Get Character Func Ptr   */
        ap_putc,                          	/* Application Put Character Func Ptr   */
        (PIL_UINT32) cnt_buffering_size,	/* Content data buffering size          */
        (PIL_INT32) 3,						/* How many appl spec items to process  */
        asi_names                           /* ASI names 							*/
    );

						   
	if (err != PIL_OK) 
	{
        /* ----------------------------------------- */
        /* pil_pg_init failed, report error and quit */
        /* ----------------------------------------- */
		error_report (err);
		keep_parsing = PIL_FALSE;
	}
	else 
	{
        /* ------------------------------------------------ */
        /* pil_pg_init completed successfully.              */
        /* Parse until end of file or an error 				*/
        /* ------------------------------------------------ */
		keep_parsing = PIL_TRUE;
	}
	

	while (keep_parsing) 
	{
		/* ---------------------------------------------------------------- */
		/* Reset the PIL buffer back to the beginning to make maximum use	*/
		/* of the allocated memory. Then invoke the parser to get the next	*/
		/* PIL component.													*/
		/* ---------------------------------------------------------------- */
		pil_set_buffer (pilbuf, (PIL_UINT32) pil_buffer_size);
		err = pil_get_component (&pc);


		/* --------------------------------- */
		/* Process the component accordingly */
		/* --------------------------------- */
		switch (err) 
		{
			case PIL_OK:
			{
				getstats (pc);
				continue;
			}
			case PIL_OUT_OF_MEMORY:
			{
				error_report (err);
				printf("Buffer size was %lu bytes.\n",pil_buffer_size);
				if (restart_parser() == -1)
				{
					printf ("Cannot restart the parser\n");
					keep_parsing = PIL_FALSE;
					continue;
				}
				printf ("Restarting parser...\n");
				printf ("New PIL buffer size: %lu bytes...\n\n", pil_buffer_size);
				clearstats ();
				continue;
			}
			case PIL_END_OF_FILE:
			{
				printstats();
				keep_parsing = PIL_FALSE;
				continue;
			}
			default:
			{
				keep_parsing = PIL_FALSE;
				printf("Error encountered parsing %s, line %lu\n",
				input,pil_get_line_num());
				error_report(err);
				continue;
			}
		}
	}	

	
	/* -------------------------------------------------------- */
	/* Clean up the application: close files and release memory */
	/* -------------------------------------------------------- */
	ap_cleanup();
}
