/* ----------------------------------------------------------------------------	*/
/*	Originators:																*/
/*  	Tom Arnold,	Quark, Inc.,	300 S. Jackson, 	Denver, 	CO 80209	*/
/*		Ron Perry, 	Atex, Inc.,		165 Lexington Road,	Billerica, 	MA 01821	*/
/* ----------------------------------------------------------------------------	*/
/*	The source code in this file is provided by Quark and Atex for use in the	*/
/*	development of a Publishing Interchange Language. The code provided herein	*/
/*	is non-proprietary, non-copyrighted, and is for use in the public domain.	*/
/*	The source code contained herein is provided "as is".						*/
/* ----------------------------------------------------------------------------	*/
/*	QUARK AND ATEX DISCLAIM ALL WARRANTIES, EXPRESSED OR IMPLIED, WITH REGARD	*/ 
/*	TO THE ENCLOSED SOURCE CODE. QUARK AND ATEX DISCLAIM ANY IMPLIED WARRANTY,	*/
/*	INCLUDING, BUT NOT LIMITED TO ANY IMPLIED WARRANTY OF FITNESS FOR A			*/
/*	PARTICULAR PURPOSE OR MERCHANTABILITY.										*/
/* ----------------------------------------------------------------------------	*/

/*$au ------------------------------ audits -----------------------------------
    Authors:Tom Arnold                          Ron Perry
            Quark, Inc.                         Atex, Inc.
            300 South Jackson                   165 Lexington Road
            Denver, Colorado 80209              Billerica, MA   01821
            (303) 934-2211                      (508) 670-7259

    Ver     Date       Who  	Etc
    v01.02  24-apr-91  tla,rp	Update for PIL 5.0
    v01.01  03-apr-91  tla		Added audit section. Fixed prototype for 
    							pil_put_color
---------------------------------------------------------------------------- */

/* ------------------------------------------------------------------------	*/
/*		P I L  P U B L I C  S E C T I O N									*/
/* ------------------------------------------------------------------------	*/


/* ----------------------------------------------- */
/* Basic Data Types - modify to suit your platform */
/* ----------------------------------------------- */


typedef		char			PIL_INT8;
typedef		int				PIL_INT16;
typedef		long int		PIL_INT32;
typedef		unsigned char	PIL_UINT8;
typedef		unsigned int	PIL_UINT16;
typedef		unsigned long	PIL_UINT32;
typedef		double			PIL_DOUBLE;
typedef		int				PIL_BOOL;


/* --------------------------------------------------------------------- */
/* To employ ANSI style function prototypes #define PIL_ANSI_PROTOTYPES. */
/* No #define results in K&R function declarations.						 */
/* --------------------------------------------------------------------- */


#define		PIL_ANSI_PROTOTYPES


/* --------------------------------------------------------------------------- *\
	Pointer defintions - These are used to allow pointers to be declared as
	near or far, for environments with segemented memory models (e.g., ms-dos,
	windows 3.x) For plain ms-dos, leaving these #defined to nothing should
	work because the compiler provides the proper default for the memory model
	being used.
	For Windows, include a #define WINDOWS3 statement here, which will force
	pointers to be FAR.
	
	Pointers to functions should include FPTRTYPE before the *.
	All other pointer declarations should include PTRTYPE before the *, and all
	pointers to pointers should use PTRPTR.
	Also, it is sometimes convienient to get rid of register declarations for
	pointers. The #define FAST can be used to define it or get rid of it.
\* --------------------------------------------------------------------------- */


#ifdef 		WINDOWS3
#define 	FPTRTYPE 	far
#define 	PTRTYPE 	far
#define 	PTRPTR 		PTRTYPE * PTRTYPE *
#define 	FAST		
#else
#define 	FPTRTYPE 
#define 	PTRTYPE 
#define 	PTRPTR 		PTRTYPE * PTRTYPE *
#define 	FAST		register
#endif


/* ----------------------------- */
/* Defines for common constructs */
/* ----------------------------- */


#define		PIL_TRUE	(PIL_BOOL) 1
#define		PIL_FALSE	(PIL_BOOL) 0
#ifndef 	NULL
#define		NULL 		0L
#endif 


/* --------------------------------------------------------------------------- *\
   Declare MEMORY_ALIGN_SIZE. This is used for processors that do not allow 
   structures or pointers to start on random addresses. The pil memory
   "allocater", pil_alloc, will ensure that all memory chunks it returns start
   on the proper boundary, i.e., ADDRESS % MEMORY_ALIGN_SIZE == 0
   For example, a DEC PDP-11 requires that pointers and 16-bit integers
   start on an even address, so MEMORY_ALIGN_SIZE should be 2.
   A Motorola 68000 wants alignment on 4-byte boundaries, but a 68020 doesn't
   care. Therefore, if you might be running on a 68000 use 4 for 
   MEMORY_ALIGN_SIZE, if you know you'll be running on 680x0, (where x > 0)
   you can comment out the #define altogether.
   
   Declare PIL_ADDR_T - This should be a typedef that gives an integral type
   that is the same size as a pointer on the given platform. It is used along
   with PIL_MEM_ALIGN_SIZE, in pil_set_buffer.
\* --------------------------------------------------------------------------- */
   
   
#define PIL_MEM_ALIGN_SIZE	(sizeof(int))
typedef PIL_UINT32			PIL_ADDR_T;

/* --------------------------------------------------------------------------- *\
   Declare PIL_ERROR, the error code type returned by all pil functions
   
   The application is responsible for translating the codes (e.g. to a 
   textual error message) and reporting the errors to the user; the pil 
   parser/generator does not do that.
\* --------------------------------------------------------------------------- */
   
   
typedef	PIL_UINT16			PIL_ERROR;			/* PIL_ERROR type definition 				*/
#define PIL_OK				(PIL_ERROR) 0		/* function completed without error 		*/
#define	PIL_OUT_OF_MEMORY	(PIL_ERROR) 1		/* not enough room left in pil buffer 		*/
#define PIL_END_OF_FILE		(PIL_ERROR) 2		/* end of file was encountered on a read 	*/
#define	PIL_NO_BUFPTR		(PIL_ERROR) 3		/* the pil buffer was not provided 			*/
#define	PIL_NO_ASALISTPTR	(PIL_ERROR) 4		/* asa list was expected, but not provided 	*/
#define	PIL_BAD_STRUCT		(PIL_ERROR) 5		/* bad structure found when generating		*/
#define	PIL_BAD_SYNTAX		(PIL_ERROR) 6		/* parser encountered illegal input 		*/
#define	PIL_BAD_FILE_TYPE	(PIL_ERROR) 7		/* input to the parser is NOT a PIL file 	*/
#define	PIL_PREV_ERR		(PIL_ERROR) 8		/* previous error encountered by parser 	*/
#define	PIL_BAD_CBUF_SIZE	(PIL_ERROR) 9		/* invalid (zero) content buffering size 	*/
#define	PIL_BAD_CMPNT_TYPE	(PIL_ERROR) 10		/* invalid component type 					*/
#define	PIL_BAD_VALUE_TYPE	(PIL_ERROR) 11		/* invalid pil_value type 					*/
#define	PIL_NO_RW_FUNCS		(PIL_ERROR) 12		/* both getc and putc func ptrs are null 	*/
#define	PIL_NO_R_FUNC		(PIL_ERROR) 13		/* getc function pointer is null 			*/
#define	PIL_NO_W_FUNC		(PIL_ERROR) 14		/* putc function pointer is null 			*/
#define	PIL_UNKN_HDR_BORD	(PIL_ERROR) 15		/* unknown byte order in cnt data hdr 		*/
#define PIL_BAD_UNITS		(PIL_ERROR) 16		/* Invalid Units (canvas or object)			*/
#define PIL_ZERO_DIMENSION	(PIL_ERROR) 17		/* Object or Canvas with zero dimension 	*/
#define PIL_BAD_NTE			(PIL_ERROR) 18		/* A name table entry is missing a field 	*/
#define PIL_OPEN_CLIPPER	(PIL_ERROR) 19		/* An open shape can't be used as a clipper */
#define PIL_ZERO_SHAPE		(PIL_ERROR) 20		/* A shape can't have zero dimension 		*/
#define PIL_UNKN_SHAPE		(PIL_ERROR) 21		/* Unknown shape 							*/
#define PIL_UNKN_PATHPT		(PIL_ERROR) 22		/* Unknown path point type 					*/
#define PIL_ZERO_LENGTH		(PIL_ERROR) 23		/* A line can't have zero length 			*/
#define PIL_NEED_MOVE_TO	(PIL_ERROR) 24		/* Path must start with a move-to 			*/
#define PIL_OPEN_CONTAINER	(PIL_ERROR) 25		/* A container must be a closed shape 		*/
#define PIL_UNKN_CM			(PIL_ERROR) 26		/* An unknown color model 					*/
#define PIL_UNKN_RENDEROP	(PIL_ERROR) 27		/* An unknown render operation 				*/
#define PIL_UNKN_FILLRULE	(PIL_ERROR) 28		/* An unknown fill rule 					*/
#define PIL_UNKN_LINEPOS	(PIL_ERROR) 29		/* Invalid Line position 					*/
#define PIL_BAD_OBJID		(PIL_ERROR) 30		/* object id list with null id pointer 		*/
#define PIL_BAD_MITERLIM	(PIL_ERROR) 31		/* Miter limit must be >= 1.0 				*/
#define PIL_BAD_COLOR		(PIL_ERROR) 32		/* Color value out of range 				*/
#define PIL_BAD_HEADER		(PIL_ERROR) 33		/* incorrect values in a cnt header 		*/
#define	PIL_BAD_ARGS		(PIL_ERROR) 34		/* invalid arguments passed to a pil func 	*/
#define PIL_WRONG_CMPNT		(PIL_ERROR) 35		/* incorrect component type provided 		*/
#define	PIL_UNKN_HDR_ENC	(PIL_ERROR) 36		/* unknown data encoding in cnt hdr 		*/


/* --------------------------------------------------------------------------- *\
   Component types - used in the type field of all pil_components
   
   These are declared as #define statements instead of an enumerated type (enum)
   because support for enums is somewhat less common and is compiler specific.
\* --------------------------------------------------------------------------- */


typedef	PIL_UINT16				PIL_COMP_TYPE;
#define PIL_LAYOUT_START_C		(PIL_COMP_TYPE) 0
#define	PIL_NAME_TABLE_C 		(PIL_COMP_TYPE) 1
#define	PIL_CANVAS_C			(PIL_COMP_TYPE) 2
#define	PIL_OBJECT_C 			(PIL_COMP_TYPE) 3
#define	PIL_TEXT_FLOW_C 		(PIL_COMP_TYPE) 4
#define	PIL_GROUP_C 			(PIL_COMP_TYPE) 5
#define	PIL_LAYOUT_END_C 		(PIL_COMP_TYPE) 6
#define PIL_CONTENT_START_C		(PIL_COMP_TYPE) 7
#define	PIL_CONTENT_HDR_C		(PIL_COMP_TYPE) 8
#define	PIL_CONTENT_DATA_C		(PIL_COMP_TYPE) 9
#define	PIL_CONTENT_END_C 		(PIL_COMP_TYPE) 10


/* --------------------------------------------------------------------------- *\
	Check to see if a pil_component c is of the desired type, t
\* --------------------------------------------------------------------------- */


#define PIL_RIGHT_CMPNT(c,t)	(((pil_component PTRTYPE *)(c))->type == t ? 1 : 0)


/* --------------------------------------------------------------------------- *\
	NameTable Types - content types used in pil_name_table_entry
\* --------------------------------------------------------------------------- */


typedef		PIL_UINT16					PIL_NT_TYPE;
#define		PIL_CNT_OTHER				(PIL_NT_TYPE) 0
#define		PIL_CNT_UNKNOWN				(PIL_NT_TYPE) 1
#define		PIL_CNT_ASCII_TEXT			(PIL_NT_TYPE) 2
#define		PIL_CNT_ATEX_ARIS			(PIL_NT_TYPE) 3
#define		PIL_CNT_ATEX_ITF			(PIL_NT_TYPE) 4
#define		PIL_CNT_EPS					(PIL_NT_TYPE) 5
#define		PIL_CNT_ICL					(PIL_NT_TYPE) 6
#define		PIL_CNT_IT8_1				(PIL_NT_TYPE) 7
#define		PIL_CNT_IT8_2				(PIL_NT_TYPE) 8
#define		PIL_CNT_MS_RTF				(PIL_NT_TYPE) 9
#define		PIL_CNT_MS_WORD_3_0			(PIL_NT_TYPE) 10
#define		PIL_CNT_MS_WORD_4_0			(PIL_NT_TYPE) 11
#define		PIL_CNT_MIF					(PIL_NT_TYPE) 12
#define		PIL_CNT_PICT				(PIL_NT_TYPE) 13
#define		PIL_CNT_PIL_LAYOUT			(PIL_NT_TYPE) 14
#define		PIL_CNT_POSTSCRIPT			(PIL_NT_TYPE) 15
#define		PIL_CNT_QUARK_XP_TAGS		(PIL_NT_TYPE) 16
#define		PIL_CNT_SCITEX_CT			(PIL_NT_TYPE) 17
#define		PIL_CNT_SCITEX_HS			(PIL_NT_TYPE) 18
#define		PIL_CNT_SCITEX_LW			(PIL_NT_TYPE) 19
#define		PIL_CNT_SGML				(PIL_NT_TYPE) 20
#define		PIL_CNT_TIFF				(PIL_NT_TYPE) 21
#define		PIL_CNT_WP_1_00				(PIL_NT_TYPE) 22
#define		PIL_CNT_WP_1_02				(PIL_NT_TYPE) 23
#define		PIL_CNT_XYQUEST_XYWRITE		(PIL_NT_TYPE) 24


/* --------------------------------------------------------------------------- *\
	NameTable Domains - domain types used in pil_name_table_entry
\* --------------------------------------------------------------------------- */


typedef		PIL_UINT16					PIL_NT_DMN;
#define		PIL_DMN_OTHER				(PIL_NT_DMN) 0
#define		PIL_DMN_MAC_FILE			(PIL_NT_DMN) 1
#define		PIL_DMN_UNIX_FILE			(PIL_NT_DMN) 2
#define		PIL_DMN_MSDOS_FILE			(PIL_NT_DMN) 3
#define		PIL_DMN_OS2_FILE			(PIL_NT_DMN) 4
#define		PIL_DMN_VMS_FILE			(PIL_NT_DMN) 5
#define		PIL_DMN_VM_FILE				(PIL_NT_DMN) 6
#define		PIL_DMN_LCD_FILE			(PIL_NT_DMN) 7
#define		PIL_DMN_INLINE				(PIL_NT_DMN) 8
#define		PIL_DMN_UNKNOWN				(PIL_NT_DMN) 9


/* --------------------------------------------------------------------------- *\
	NameTable Values - value types used in pil_name_table_entry
\* --------------------------------------------------------------------------- */


typedef		PIL_UINT16					PIL_NT_VAL;
#define		PIL_VAL_OTHER				(PIL_NT_VAL) 0
#define		PIL_VAL_INLINE				(PIL_NT_VAL) 1
#define		PIL_VAL_UNKNOWN				(PIL_NT_VAL) 2


/* --------------------------------------------------------------------------- *\
	Application Specific Value Types
\* --------------------------------------------------------------------------- */


typedef		PIL_UINT16					PIL_ASA_VAL_TYPE;
#define		PIL_INT8_CODE				(PIL_ASA_VAL_TYPE) 0x0000
#define		PIL_INT16_CODE				(PIL_ASA_VAL_TYPE) 0x0001
#define		PIL_INT32_CODE				(PIL_ASA_VAL_TYPE) 0x0002
#define		PIL_UINT8_CODE				(PIL_ASA_VAL_TYPE) 0x0003
#define		PIL_UINT16_CODE				(PIL_ASA_VAL_TYPE) 0x0004
#define		PIL_UINT32_CODE				(PIL_ASA_VAL_TYPE) 0x0005
#define		PIL_DOUBLE_CODE				(PIL_ASA_VAL_TYPE) 0x0006
#define		PIL_STRING_CODE				(PIL_ASA_VAL_TYPE) 0x0007
#define		PIL_INT8_LIST_CODE			(PIL_ASA_VAL_TYPE) 0x1000
#define		PIL_INT16_LIST_CODE			(PIL_ASA_VAL_TYPE) 0x1001
#define		PIL_INT32_LIST_CODE			(PIL_ASA_VAL_TYPE) 0x1002
#define		PIL_UINT8_LIST_CODE			(PIL_ASA_VAL_TYPE) 0x1003
#define		PIL_UINT16_LIST_CODE		(PIL_ASA_VAL_TYPE) 0x1004
#define		PIL_UINT32_LIST_CODE		(PIL_ASA_VAL_TYPE) 0x1005
#define		PIL_DOUBLE_LIST_CODE		(PIL_ASA_VAL_TYPE) 0x1006
#define		PIL_STRING_LIST_CODE		(PIL_ASA_VAL_TYPE) 0x1007
#define		PIL_LIST_CODE_OFFSET		(PIL_ASA_VAL_TYPE) 0x1000
#define		IS_PIL_VALUE_LIST(val)		((val) & (PIL_ASA_VAL_TYPE) 0x1000)


/* --------------------------------------------------------------------------- *\
	Renderable Content Types (used in pil_object structure)
\* --------------------------------------------------------------------------- */


typedef	PIL_UINT16			PIL_RC_TYPE;
#define	PIL_RC_UNKNOWN		(PIL_RC_TYPE) 0x0000
#define	PIL_RC_TEXT			(PIL_RC_TYPE) 0x0001
#define	PIL_RC_GEOMETRY		(PIL_RC_TYPE) 0x0002
#define	PIL_RC_LINEART		(PIL_RC_TYPE) 0x0004
#define	PIL_RC_IMAGE		(PIL_RC_TYPE) 0x0008


/* --------------------------------------------------------------------------- *\
	Types of path points - used to describe open and closed paths
\* --------------------------------------------------------------------------- */


typedef	PIL_UINT16		PIL_PATH_PART;
#define	PIL_MOVE_TO		(PIL_PATH_PART) 0x01
#define	PIL_LINE_TO		(PIL_PATH_PART) 0x02
#define	PIL_CURVE_TO	(PIL_PATH_PART) 0x03
#define	PIL_ARC_TO		(PIL_PATH_PART) 0x04


/* --------------------------------------------------------------------------- *\
	Open and closed primitives - used in pil_graphic, pil_clipper, and 
	pil_container.
	
	Note: The use of PIL_NO_SHAPE in the type field of the shape for the 
	clipper, graphic, or container means that the clipper, graphic, or 
	container for the object does not really exist.
\* --------------------------------------------------------------------------- */


typedef	PIL_UINT16					PIL_OPEN_PRIM;			/* Only open primitives 	*/
typedef	PIL_UINT16					PIL_CLOSED_PRIM;		/* Only closed primitives 	*/
typedef	PIL_UINT16					PIL_PRIMITIVE;			/* All primitives 			*/
#define	PIL_NO_SHAPE				(PIL_PRIMITIVE) 0x0000	/* Indicates no shape		*/
#define PIL_LINE_SHAPE				(PIL_PRIMITIVE) 0x0001
#define PIL_ARC_SHAPE				(PIL_PRIMITIVE) 0x0002
#define PIL_BEZIER_SHAPE			(PIL_PRIMITIVE) 0x0003
#define	PIL_POLYLINE_SHAPE			(PIL_PRIMITIVE) 0x0004
#define	PIL_OPENPATH_SHAPE			(PIL_PRIMITIVE) 0x0005
#define	PIL_RECTANGLE_SHAPE			(PIL_PRIMITIVE) 0x0100
#define PIL_CIRCLE_SHAPE			(PIL_PRIMITIVE) 0x0200
#define PIL_ELLIPSE_SHAPE			(PIL_PRIMITIVE) 0x0300
#define PIL_POLYGON_SHAPE			(PIL_PRIMITIVE) 0x0400
#define PIL_CLOSEDPATH_SHAPE		(PIL_PRIMITIVE) 0x0500
#define PIL_CLIPPER_SHAPE			(PIL_PRIMITIVE) 0x0600
#define PIL_CONTAINER_SHAPE			(PIL_PRIMITIVE) 0x0700
#define PIL_OPEN_PRIMITIVE_MASK		(PIL_PRIMITIVE) 0x00FF
#define PIL_CLOSED_PRIMITIVE_MASK	(PIL_PRIMITIVE) 0xFF00
#define IS_OPEN_SHAPE(type)			((type) & PIL_OPEN_PRIMITIVE_MASK)
#define IS_CLOSED_SHAPE(type) 		((type) & PIL_CLOSED_PRIMITIVE_MASK)


/* --------------------------------------------------------------------------- *\
	Declare flags and values used in the pil_content_hdr
\* --------------------------------------------------------------------------- */


typedef		PIL_UINT16					PIL_CNT_ORDERING;
#define		PIL_LITTLE_ENDIAN			(PIL_CNT_ORDERING) 0
#define		PIL_BIG_ENDIAN				(PIL_CNT_ORDERING) 1
typedef		PIL_UINT16					PIL_CNT_ENCODING;
#define		PIL_ASCII_CONTENT			(PIL_CNT_ENCODING) 0
#define		PIL_BINARY_CONTENT			(PIL_CNT_ENCODING) 1


/* --------------------------------------------------------------------------- *\
	Declare the definitions of colors in the various color models supported.
\* --------------------------------------------------------------------------- */


typedef		PIL_UINT16				PIL_COLOR_MODEL;
#define		PIL_DEV_GRAY_CM			(PIL_COLOR_MODEL) 0
#define		PIL_CAL_GRAY_CM			(PIL_COLOR_MODEL) 1
#define		PIL_DEV_RGB_CM			(PIL_COLOR_MODEL) 2
#define		PIL_CAL_RGB_CM			(PIL_COLOR_MODEL) 3
#define		PIL_DEV_CMYK_CM			(PIL_COLOR_MODEL) 4
#define		PIL_PANTONE_CM			(PIL_COLOR_MODEL) 5
#define		PIL_SPOT_CM				(PIL_COLOR_MODEL) 6


/* --------------------------------------------------------------------------- *\
	Fill rules - used in the pil_graphic to identify the algorithm used
	to find the inside and outside of closed shapes.
\* --------------------------------------------------------------------------- */


typedef	PIL_UINT16				PIL_FILL_RULE;
#define	PIL_EVEN_ODD			(PIL_FILL_RULE) 0x01
#define	PIL_NON_ZERO_WINDING	(PIL_FILL_RULE) 0x02


/* --------------------------------------------------------------------------- *\
	Render Operations - used in the pil_graphic to declare whether it should 
	be filled, have its stroke (outline) drawn, both, or neither.
\* --------------------------------------------------------------------------- */


typedef	PIL_UINT16					PIL_RENDER_OP;
#define	PIL_NO_RENDER_OP			(PIL_RENDER_OP) 0x0000
#define	PIL_FILL_RENDER_OP			(PIL_RENDER_OP) 0x0001
#define	PIL_STROKE_RENDER_OP		(PIL_RENDER_OP) 0x0002
#define	PIL_FILL_STROKE_RENDER_OP	(PIL_RENDER_OP) 0x0003
#define	PIL_IS_FILL_OP(ARG)			(ARG & PIL_FILL_RENDER_OP)
#define	PIL_IS_STROKE_OP(ARG)		(ARG & PIL_STROKE_RENDER_OP)


/* --------------------------------------------------------------------------- *\
	Declare definitions used for the lineposition field of the pil_graphic
\* --------------------------------------------------------------------------- */


typedef	PIL_UINT16			PIL_LINE_POSITION;
#define	PIL_CENTERED		(PIL_LINE_POSITION) 0x00
#define PIL_INSIDE			(PIL_LINE_POSITION) 0x01
#define PIL_OUTSIDE			(PIL_LINE_POSITION) 0x02


/* --------------------------------------------------------------------------- *\
	Declare definitions used for the linecap field of the pil_graphic
\* --------------------------------------------------------------------------- */


typedef	PIL_UINT16			PIL_CAP_TYPE;
#define	PIL_BUTT_CAP		(PIL_CAP_TYPE) 0x00
#define PIL_ROUND_CAP		(PIL_CAP_TYPE) 0x01
#define PIL_PROJECTING_CAP	(PIL_CAP_TYPE) 0x02


/* --------------------------------------------------------------------------- *\
	Declare definitions used for the linejoin field of the pil_graphic
\* --------------------------------------------------------------------------- */


typedef	PIL_UINT16			PIL_JOIN_TYPE;
#define	PIL_MITER_JOIN		(PIL_JOIN_TYPE) 0x00
#define PIL_BEVEL_JOIN		(PIL_JOIN_TYPE) 0x01
#define PIL_ROUND_JOIN		(PIL_JOIN_TYPE) 0x02


/* --------------------------------------------------------------------------- *\
	Declare definitions used for the object type of the pil_object
\* --------------------------------------------------------------------------- */


typedef	PIL_UINT16			PIL_OBJ_TYPE;
#define	PIL_OBJ_UNKNOWN		(PIL_OBJ_TYPE) 0x00
#define	PIL_OBJ_TEXT		(PIL_OBJ_TYPE) 0x01
#define PIL_OBJ_PICTURE		(PIL_OBJ_TYPE) 0x02
#define PIL_OBJ_PRIMITIVE	(PIL_OBJ_TYPE) 0x03


/* --------------------------------------------------------------------------- *\
	Define the PIL Language defaults for all non-pointer structure elements.
	Pointer elements are initialized to NULL as specified in the API
	documentation.
\* --------------------------------------------------------------------------- */


#define		PIL_DFT_CNV_CLIPPER			PIL_NO_SHAPE
#define		PIL_DFT_OBJ_UNITS			(PIL_DOUBLE) 0
#define		PIL_DFT_OBJ_ROTANG			(PIL_DOUBLE) 0
#define		PIL_DFT_OBJ_ROTPT_X			(PIL_INT32) 0
#define		PIL_DFT_OBJ_ROTPT_Y			(PIL_INT32) 0
#define		PIL_DFT_OBJ_CLIPPER			PIL_NO_SHAPE
#define		PIL_DFT_OBJ_CONTAINER		PIL_NO_SHAPE
#define		PIL_DFT_OBJ_GRAPHIC			PIL_NO_SHAPE
#define		PIL_DFT_OBJ_SRCRECT_X		(PIL_DOUBLE) 0
#define		PIL_DFT_OBJ_SRCRECT_Y		(PIL_DOUBLE) 0
#define		PIL_DFT_OBJ_SRCRECT_W		(PIL_DOUBLE) 0
#define		PIL_DFT_OBJ_SRCRECT_H		(PIL_DOUBLE) 0
#define		PIL_DFT_OBJ_RCTYPE			PIL_RC_UNKNOWN
#define		PIL_DFT_OBJ_TYPE			PIL_OBJ_UNKNOWN
#define		PIL_DFT_FILL_RULE			PIL_EVEN_ODD
#define		PIL_DFT_LINE_CAP			PIL_BUTT_CAP
#define		PIL_DFT_LINE_JOIN			PIL_MITER_JOIN
#define		PIL_DFT_LINE_POSITION		PIL_CENTERED
#define		PIL_DFT_LINE_WIDTH			(PIL_UINT32) 10
#define		PIL_DFT_MITER_LIMIT			(PIL_DOUBLE) 10
#define		PIL_DFT_CNT_ESC				(PIL_UINT8) 93
#define		PIL_DFT_CNT_BEG				(PIL_UINT8) 123
#define		PIL_DFT_CNT_END				(PIL_UINT8) 125


/* --------------------------------------------------------------------------- *\
	Declare the pil_point data structure, and numerous shape data structures
\* --------------------------------------------------------------------------- */


typedef struct pp 
{
	PIL_INT32 x;
	PIL_INT32 y;
} 	pil_point;

#define SAMEPOINT(a,b)	((a).x == (b).x && (a).y == (b).y)

typedef struct pdp 
{
	PIL_DOUBLE	x;
	PIL_DOUBLE	y;
} 	pil_dbl_point;


typedef struct pline 
{
	pil_point	start;
	pil_point	end;
} 	pil_line;


typedef struct pbezier 
{
	pil_point	start;
	pil_point	start_ctrl_pt;
	pil_point	end_ctrl_pt;
	pil_point	end;
} 	pil_bezier;


typedef struct pr 
{
	pil_point	ul;
	PIL_UINT32	width;
	PIL_UINT32	height;
} 	pil_rectangle;


typedef struct pdr 
{
	pil_dbl_point	ul;
	PIL_DOUBLE		width;
	PIL_DOUBLE		height;
} 	pil_dbl_rect;


typedef struct psquare 
{
	pil_point	ul;
	PIL_UINT32	length;
}	pil_square;


typedef struct parc 
{
	pil_square	arc_square;
	PIL_DOUBLE	start_angle;
	PIL_DOUBLE	end_angle;
} 	pil_arc;


typedef pil_rectangle 	pil_ellipse;
typedef	pil_square 		pil_circle;


/* --------------------------------------------------------------------------- *\
	Each point on a path is one of the following types:
	PIL_MOVE_TO, PIL_LINE_TO, PIL_CURVE_TO, PIL_ARC_TO.
	A pil_path is a linked list of these points.
\* --------------------------------------------------------------------------- */


typedef struct ppathpt 
{
	PIL_PATH_PART			type;		/* PIL_MOVE_TO, PIL_LINE_TO, ... */
	struct ppathpt PTRTYPE 	*next;		/* Pointer to next list element */
	union 								/* Union of MoveTo, LineTo, ... */
	{
		struct pmt 
		{
			pil_point	pt;				/* Move to pt */
		} 	mt;							/* Move to */
		struct plt 
		{
			pil_point	pt;				/* Draw line to pt */
		} 	lt;							/* Line to */
		struct pct 
		{
			pil_point	ctrl_pt_1;		/* Starting Point's control point */
			pil_point	ctrl_pt_2;		/* Ending Point's control point */
			pil_point	end_pt;			/* End point */
		} 	ct;							/* Curve to */
		struct pat 
		{
			pil_point	ctrl_pt_1;		/* Starting Point's control point */
			pil_point	ctrl_pt_2;		/* Ending Point's control point */
			PIL_INT32	radius;			/* Radius as defined in PostScript RedBook */
		} 	at;							/* Arc to */
	}	p;
} 	pil_path_pt;
	
typedef pil_path_pt	pil_path;


/* --------------------------------------------------------------------------- *\
	Open paths may not be used for clipping shapes, nor for container shapes.
	pil_open_path and pil_closed_path are provided just to make this distinction
	clear in the declaration of clipper, containers and graphics below.
\* --------------------------------------------------------------------------- */


typedef pil_path	pil_open_path;
typedef pil_path	pil_closed_path;


/* --------------------------------------------------------------------------- *\
	All the points in a pil_polygon or pil_polyline are the same.
	
	Polylines may not be used for clipping shapes, nor for container shapes.
	pil_polyline and pil_polygon are provided just to make this distinction
	clear in the declaration of clipper, containers and graphics below.
\* --------------------------------------------------------------------------- */


typedef struct ppolypt 
{
	struct ppolypt PTRTYPE 	*next;
	pil_point				p;
}	pil_poly_pt;


typedef	pil_poly_pt	pil_polygon;	/* A polygon is a list of points 			*/
typedef pil_poly_pt	pil_polyline;	/* A polygonal line is a list of points too	*/


typedef struct pcprimitive 
{
	PIL_CLOSED_PRIM		type;		/* shape type (PIL_RECTANGLE_SHAPE, etc.) */
	union 
	{
		pil_rectangle	rect;
		pil_circle		circle;
		pil_ellipse		ellipse;
		pil_polygon		polygon;
		pil_closed_path	closedpath;
	} 	shape;
} 	pil_closed_primitive;


/* --------------------------------------------------------------------------- *\
	The pil_clipper structure is used to clip the content of a canvas or object.
	The pil_container structure is used to provide a shape to flow text into.
	It is part of the pil_object structure.
\* --------------------------------------------------------------------------- */


typedef pil_closed_primitive pil_clipper;
typedef pil_closed_primitive pil_container;


/* --------------------------------------------------------------------------- *\
	Used in the definition of pil_graphic.
\* --------------------------------------------------------------------------- */


typedef struct pprimitive 
{
	PIL_PRIMITIVE		type;		/* shape type (PIL_RECTANGLE_SHAPE, etc.) */
	union 
	{
		pil_line		line;
		pil_arc			arc;
		pil_bezier		bezier;
		pil_polyline	polyline;
		pil_open_path	openpath;
		pil_rectangle	rect;
		pil_circle		circle;
		pil_ellipse		ellipse;
		pil_polygon		polygon;
		pil_closed_path	closedpath;
	} 	shape;
}	pil_primitive;


/* --------------------------------------------------------------------------- *\
   Declare pil_value, the value-data linked-list structure type

   Pointed to by a pil_asa structure and the previous
   pil_value structure, if any.
\* --------------------------------------------------------------------------- */


typedef struct  pil_val 
{
    struct pil_val PTRTYPE *next ;	/* ptr to next value in the list 		*/
    union 							/* Union of possible data types			*/	
    {
        double			dbl ;		/* double precision floating data value */
        PIL_INT8		int8 ;		/*  8 bit integer data value            */
        PIL_INT16		int16 ;		/* 16 bit integer data value            */
        PIL_INT32		int32 ;		/* 32 bit integer data value            */
        PIL_UINT8		uint8 ;		/*  8 bit unsigned integer data value   */
        PIL_UINT16		uint16 ;	/* 16 bit unsigned integer data value   */
        PIL_UINT32		uint32 ;	/* 32 bit unsigned integer data value   */
        char PTRTYPE 	*string ;	/* string data-value pointer        	*/
    } 	data;
} 	pil_value ;


/* --------------------------------------------------------------------------- *\
	Declare pil_asa, the application-specific data
	linked-list structure type.
	
	Pointed-to by a pil_asi structure and by the previous
	pil_asa structure, if any.
\* --------------------------------------------------------------------------- */


typedef struct  pil_appl_specific_attr 
{
    struct pil_appl_specific_attr  
    PTRTYPE *next ;  						/* next attribute in the list 		*/
    pil_value PTRTYPE	*value ;            /* pointer to the value	list		*/
    PIL_UINT32			count ;       		/* # of values in the value list 	*/
    PIL_ASA_VAL_TYPE	type ;      		/* type of each attribute value:	*
                                             * PIL_INT8_CODE;					*
                                             * PIL_INT16_CODE;					*
                                             * etc...							*/
    char PTRTYPE 		*attr_name ;        /* attribute name					*/
} 	pil_asa ;


/* --------------------------------------------------------------------------- *\
   Declare pil_asi the application-specific-item linked-list
   structure type.

   Pointed-to by a pil_canvas, or pil_object structure and
   by the previous pil_asi structure, if any.
\* --------------------------------------------------------------------------- */


typedef struct  pil_appl_specific_item 
{
    struct pil_appl_specific_item   
    PTRTYPE 		*next ; 				/* next item in the list 			*/
	pil_asa PTRTYPE	*asa ;  				/* Pointer to a list of structures 	*
											 * containing attributes        	*/
    char PTRTYPE 	*asi_name ;            	/* ApplicationName-ExtensionName	*/
} 	pil_asi ;


/* --------------------------------------------------------------------------- *\
	Declare data structures representing the PIL notion of color
\* --------------------------------------------------------------------------- */


typedef struct pdevgray
{
	PIL_DOUBLE value;
} 	pil_device_gray_color;


typedef struct pdevrgb 
{
	PIL_DOUBLE	r;
	PIL_DOUBLE	g;
	PIL_DOUBLE	b;
} 	pil_device_rgb_color;


typedef struct pdevcmyk 
{
	PIL_DOUBLE	c;
	PIL_DOUBLE	m;
	PIL_DOUBLE	y;
	PIL_DOUBLE	k;
} 	pil_device_cmyk_color;


typedef pil_device_gray_color	pil_calibrated_gray_color;
typedef pil_device_rgb_color	pil_calibrated_rgb_color;


typedef struct ppantone 
{
	char PTRTYPE *name;
} 	pil_pantone_color;


typedef struct pspot 
{
	char PTRTYPE *name;
}	pil_spot_color;


typedef union pcolor 
{
	pil_device_gray_color		devgray;
	pil_calibrated_gray_color	calgray;
	pil_device_rgb_color		devrgb;
	pil_calibrated_rgb_color	calrgb;
	pil_device_cmyk_color		devcmyk;
	pil_pantone_color			pantone;
	pil_spot_color				spot;
} 	pil_color;


/* --------------------------------------------------------------------------- *\
	The pil_graphic structure provides a way to draw simple geometric shapes as
	part of an object, without having to create a separate content file to hold
	the "renderable" content. This could be used in conjunction with a separate
	content file, for example: the renderable content of an object could be an
	image file, and the object graphic could draw a border around it.
	
	Or the object graphic could convey the entire visible representation of the 
	object, for example a line drawn on a page.
	
	The pil_graphic is part of the pil_object structure.
\* --------------------------------------------------------------------------- */


typedef struct pgraphic 
{
	PIL_COLOR_MODEL		color_model;	/* PIL_DEV_GRAY_CM, etc.			*/
	PIL_RENDER_OP		renderop;		/* PIL_FILL_RENDER_OP, etc.			*/
	PIL_FILL_RULE		fill_rule;		/* PIL_EVEN_ODD, etc.				*/
	pil_color			fill_color;		/* Fill color for fill operations	*/
	pil_color			stroke_color;	/* Stroke color for stroke ops		*/
	PIL_CAP_TYPE		linecap;		/* PIL_BUTT_CAP, etc.				*/
	PIL_JOIN_TYPE		linejoin;		/* PIL_MITER_JOIN, etc.				*/
	PIL_LINE_POSITION	lineposition;	/* PIL_CENTERED, etc.				*/
	PIL_UINT32			linewidth;		/* Line width for stroking			*/
	PIL_DOUBLE			miter_limit;	/* Miter limit for stroking			*/
	pil_primitive		shape;			/* Shape description				*/
} 	pil_graphic;


/* --------------------------------------------------------------------------- *\
	Name Table Entry Data Structure - used in pil_name_table
\* --------------------------------------------------------------------------- */


typedef struct pnametblentry 
{
	struct pnametblentry 
	PTRTYPE 		*next;					/* pointer to next name table entry			*/
	char PTRTYPE 	*name;					/* as referenced by other pil structures 	*/
	PIL_NT_TYPE		content_type_code;		/* PIL_CNT_EPS, PIL_CNT_OTHER, ...			*/
	PIL_NT_DMN		domain_type_code;		/* PIL_DMN_MAC_FILE, PIL_DMN_OTHER, ...		*/
	PIL_NT_VAL		value_type_code;		/* PIL_VAL_INLINE, PIL_VAL_/*	, ...		*/
	char PTRTYPE 	*content_type_string;	/* content type in string format			*/
	char PTRTYPE 	*domain_type_string;	/* domain type in string format				*/
	char PTRTYPE	*value_type_string;		/* value type in string format				*/
} 	pil_name_table_entry;


/* --------------------------------------------------------------------------- *\
	Object Id List Data Structure - used in pil_text_flow and pil_group
\* --------------------------------------------------------------------------- */


typedef	struct  pobjectidlst 
{
	struct pobjectidlst PTRTYPE	*next;	/* Pointer to next id in list */
	char PTRTYPE 				*id;	/* Actual id */
} 	pil_object_id_list;


/* --------------------------------------------------------------------------- *\
	The pil_component is an abstraction created for the convience of the 
	application program interface. There is no pil_component defined within the
	syntax of the PIL. The pil_component structure provides a generalized way
	for the application and the parser or generator to pass various pil 
	structures back and forth.
\* --------------------------------------------------------------------------- */


typedef union pcomponent 
{
	PIL_COMP_TYPE		type;				/* Component Type: PIL_CANVAS_C, etc... */
	struct PIL_LAYOUT_START					/* Layout Start Component 				*/
	{										/* ------------------------------------ */
		PIL_COMP_TYPE	type;				/* Component Type: PIL_LAYOUT_START_C 	*/			
		char PTRTYPE 	*version;			/* Example: pil-layout-4.0				*/
		char PTRTYPE 	*name;				/* Name of layout entity				*/
	} 	ls;									/* Union name							*/
	struct PIL_NAME_TABLE 					/* Name Table Component					*/
	{										/* ------------------------------------ */
		PIL_COMP_TYPE	type;				/* Component Type: PIL_NAME_TABLE_C 	*/
		pil_name_table_entry 				/* Pointer to the next name... 			*/
		PTRTYPE *entries;					/* table entry (defined above) 			*/
	} 	nt;									/* Union name							*/
	struct PIL_CANVAS 						/* Canvas Component						*/
	{										/* ------------------------------------ */
		PIL_COMP_TYPE	type;				/* Component Type: PIL_CANVAS_C 		*/
		char PTRTYPE 	*username;			/* user visible name 					*/
		char PTRTYPE 	*cantype;			/* canvas type 							*/
		PIL_DOUBLE		units;				/* # of nanometers per user unit 		*/
		PIL_UINT32		width;				/* width of canvas 						*/
		PIL_UINT32		height;				/* height of canvas 					*/
		pil_clipper		clipshape;			/* canvas clipper shape 				*/
		pil_asi PTRTYPE	*asi;				/* application specific attributes 		*/
	} 	pc;									/* Union name							*/
	struct PIL_OBJECT 						/* Object Component 					*/
	{										/* ------------------------------------ */
		PIL_COMP_TYPE	type;				/* Component Type: PIL_OBJECT_C 		*/
		pil_rectangle	dest_rect;			/* (unrotated) rect where cnt is imaged */
		char PTRTYPE 	*id;				/* used by pil_text_flow and pil_group 	*/
		PIL_DOUBLE		units;				/* nanometers per user unit 			*/
		char PTRTYPE 	*username;			/* used for dummying 					*/
		PIL_OBJ_TYPE 	objtype;			/* object type 							*/
		PIL_DOUBLE		rotation_angle;		/* rotation angle 						*/
		pil_point		rotation_point;		/* rotation point 						*/
		pil_clipper		clipshape;			/* what to clip the content to 			*/
		pil_container	container;			/* container shape for text flow 		*/
		pil_graphic		graphic;			/* graphic for simple rules, etc. 		*/
		pil_dbl_rect	src_rect;			/* source rectangle of content 			*/
		char PTRTYPE 	*rcname;			/* renderable content name 				*/
		PIL_RC_TYPE		rctype;				/* PIL_RC_TEXT, etc... 					*/
		pil_asi PTRTYPE	*asi;				/* application specific items 			*/
	} 	po;									/* Union name							*/
	struct  PIL_TEXT_FLOW 					/* Text Flow Component					*/
	{										/* ------------------------------------ */
		PIL_COMP_TYPE	type;				/* Component Type: PIL_TEXT_FLOW_C 		*/
		char PTRTYPE 	*label;				/* label to identify this pil_text_flow	*/
		char PTRTYPE 	*from_layout;		/* layout the text flows from 			*/
		char PTRTYPE 	*from_label;		/* text flow the text flows from 		*/
		char PTRTYPE 	*to_layout;			/* layout the text flows to 			*/
		char PTRTYPE 	*to_label;			/* text flow the text flows to 			*/
		pil_object_id_list PTRTYPE *content;/* content object(s) to get text from 	*/
		pil_object_id_list PTRTYPE *objects;/* ids of objects on this canvas,		*
								 			 * in flow sequence, to flow text into 	*/
	} 	tf;									/* Union name							*/
	struct PIL_GROUP 						/* Group Component						*/
	{										/* ------------------------------------ */
		PIL_COMP_TYPE	type;				/* Component Type: PIL_GROUP_C 			*/
		char PTRTYPE 	*name;				/* name for this group 					*/
		pil_object_id_list PTRTYPE *objects;/* ids of the objects in the group 		*/
	} 	pg;									/* Union name							*/
	struct PIL_LAYOUT_END					/* Layout End Component					*/
	{										/* ------------------------------------ */
		PIL_COMP_TYPE	type;				/* Component Type: PIL_LAYOUT_END_C 	*/
	} 	le;									/* Union name							*/
	struct PIL_CONTENT_START 				/* Content Start Component				*/
	{										/* ------------------------------------ */
		PIL_COMP_TYPE	type;				/* Component Type: PIL_CONTENT_START_C 	*/
		char PTRTYPE 	*version;			/* Example: pil-content-4.0				*/
		char PTRTYPE 	*name;				/* Name of content entity				*/
	} 	cs;									/* Union name							*/
	struct PIL_CONTENT_HDR 					/* Content Header Component				*/
	{										/* ------------------------------------ */
		PIL_COMP_TYPE		type;			/* Component Type: PIL_CONTENT_HDR_C 	*/
		PIL_CNT_ORDERING	byteorder;		/* little or big endian 				*/
		PIL_CNT_ENCODING	encoding;		/* ascii or binary						*/
		PIL_UINT8			escapechar;		/* esc. sequence char			 		*/
		PIL_UINT8			begblkchar;		/* char that was used to mark the start *
											 * of the data stream. THE BEGIN BLOCK 	*
											 * CHAR IS NOT INCLUDED IN THE data[] 	*
											 * ARRAY!								*/
		PIL_UINT8			endblkchar;		/* char that was used to mark the end 	*
											 * of the data stream. THE END BLOCK 	*
											 * CHAR IS NOT INCLUDED IN THE data[] 	*
											 * ARRAY!								*/
	} 	ch;									/* Union name							*/
	struct PIL_CONTENT_DATA					/* Content Data Component				*/
	{										/* ------------------------------------ */
		PIL_COMP_TYPE	type;				/* Component Type: PIL_CONTENT_DATA_C 	*/
		PIL_UINT32		length;				/* number of bytes stored *data 		*/
		char PTRTYPE 	*data;				/* pointer to the actual data 			*/
	} 	cd;									/* Union name							*/
	struct PIL_CONTENT_END 					/* Content End Component				*/
	{										/* ------------------------------------ */
		PIL_COMP_TYPE	type;				/* Component Type: PIL_CONTENT_END_C 	*/
	} 	ce;									/* Union name							*/
} 	pil_component;


typedef struct PIL_LAYOUT_START		pil_layout_start;
typedef struct PIL_NAME_TABLE		pil_name_table;
typedef struct PIL_CANVAS			pil_canvas;
typedef struct PIL_OBJECT			pil_object;
typedef struct PIL_TEXT_FLOW		pil_text_flow;
typedef struct PIL_GROUP			pil_group;
typedef struct PIL_LAYOUT_END		pil_layout_end;
typedef struct PIL_CONTENT_START	pil_content_start;
typedef struct PIL_CONTENT_HDR		pil_content_hdr;
typedef struct PIL_CONTENT_DATA		pil_content_data;
typedef struct PIL_CONTENT_END		pil_content_end;


/* ------------------------------------------------------------------------	*/
/*		P I L  P A R S E R  S E C T I O N									*/
/* ------------------------------------------------------------------------	*/


/* Parser States */
/* ------------- */
#define		PIL_PARSER_S_START				(PIL_UINT16) 0
#define		PIL_PARSER_S_EOF				(PIL_UINT16) 1
#define		PIL_PARSER_S_ENTITY				(PIL_UINT16) 2
#define		PIL_PARSER_S_ERROR				(PIL_UINT16) 3
#define		PIL_PARSER_S_LAYOUT_START		(PIL_UINT16) 4
#define		PIL_PARSER_S_NAMETABLE			(PIL_UINT16) 5
#define		PIL_PARSER_S_CANVAS				(PIL_UINT16) 6
#define		PIL_PARSER_S_OBJECT				(PIL_UINT16) 7
#define		PIL_PARSER_S_TEXTFLOW			(PIL_UINT16) 8
#define		PIL_PARSER_S_GROUP				(PIL_UINT16) 9
#define		PIL_PARSER_S_CNT_START			(PIL_UINT16) 10
#define		PIL_PARSER_S_CNT_HEADER			(PIL_UINT16) 11
#define		PIL_PARSER_S_CNT_DATA			(PIL_UINT16) 12
#define		PIL_PARSER_S_CNT_END			(PIL_UINT16) 13


/* Token Types */
/* ----------- */
#define 	PIL_TOKEN_SYMBOL		(PIL_UINT16) 0
#define 	PIL_TOKEN_ICONSTANT		(PIL_UINT16) 1
#define		PIL_TOKEN_FCONSTANT		(PIL_UINT16) 2
#define 	PIL_TOKEN_STRING		(PIL_UINT16) 3
#define 	PIL_TOKEN_BEG_BLOCK		(PIL_UINT16) 4
#define 	PIL_TOKEN_END_BLOCK		(PIL_UINT16) 5
#define 	PIL_TOKEN_ERROR			(PIL_UINT16) 6
#define 	PIL_TOKEN_EOF			(PIL_UINT16) 7


/* States for Lexical Analysis */
/* --------------------------- */
#define		PIL_S_START				(PIL_UINT16) 0
#define		PIL_S_SYMBOL			(PIL_UINT16) 1
#define		PIL_S_COMMENT			(PIL_UINT16) 2
#define		PIL_S_CONSTANT			(PIL_UINT16) 3
#define		PIL_S_STRING			(PIL_UINT16) 4
#define		PIL_S_WHITESPACE		(PIL_UINT16) 5				
#define		PIL_S_BEGBLOCK			(PIL_UINT16) 6
#define		PIL_S_ENDBLOCK			(PIL_UINT16) 7
#define		PIL_S_POSSCONST			(PIL_UINT16) 8
#define		PIL_S_ERROR				(PIL_UINT16) 9


/* Character definitions for Lexical Analysis */
/* ------------------------------------------ */
#define		PIL_C_LETTER			(PIL_UINT16) 0
#define		PIL_C_UNDERSCORE		(PIL_UINT16) 1
#define		PIL_C_DIGIT				(PIL_UINT16) 2
#define		PIL_C_DOT				(PIL_UINT16) 3
#define		PIL_C_PLUS				(PIL_UINT16) 4
#define		PIL_C_MINUS				(PIL_UINT16) 5
#define		PIL_C_COMMENT			(PIL_UINT16) 6
#define		PIL_C_WHITESPACE		(PIL_UINT16) 7
#define		PIL_C_GARBAGE			(PIL_UINT16) 8
#define		PIL_C_STRING			(PIL_UINT16) 9
#define		PIL_C_STAR				(PIL_UINT16) 10
#define		PIL_C_BEG_BLOCK			(PIL_UINT16) 11
#define		PIL_C_END_BLOCK			(PIL_UINT16) 12


/* Ascii content data parsing states */
/* --------------------------------- */
#define		PIL_ASC_S_BEG_SEARCH	(PIL_UINT16) 0
#define		PIL_ASC_S_FETCH_DATA	(PIL_UINT16) 1
#define		PIL_ASC_S_ESCAPE		(PIL_UINT16) 2
#define		PIL_ASC_S_PUT_DATA		(PIL_UINT16) 3
#define		PIL_ASC_S_ERROR			(PIL_UINT16) 4
#define		PIL_ASC_S_END			(PIL_UINT16) 5


/* Binary content data parsing states */
/* ---------------------------------- */
#define		PIL_BIN_S_SPIN			(PIL_UINT16) 0
#define		PIL_BIN_S_COMMENT_BEG	(PIL_UINT16) 1
#define		PIL_BIN_S_COMMENT_1		(PIL_UINT16) 2
#define		PIL_BIN_S_COMMENT_2		(PIL_UINT16) 3
#define		PIL_BIN_S_PUT_DATA		(PIL_UINT16) 4
#define		PIL_BIN_S_ERROR			(PIL_UINT16) 5
#define		PIL_BIN_S_END			(PIL_UINT16) 6


/* Limit definitions */
/* ----------------- */
#define		PIL_CHAR_CLASS_TSIZE		256
#define		PIL_MAX_TOKEN_SIZE			256
#define		PIL_INT8_MIN				(PIL_DOUBLE) -128.0
#define		PIL_INT8_MAX				(PIL_DOUBLE)  127.0
#define		PIL_INT16_MIN				(PIL_DOUBLE) -32768.0
#define		PIL_INT16_MAX				(PIL_DOUBLE)  32767.0
#define		PIL_INT32_MIN				(PIL_DOUBLE) -2147483648.0
#define		PIL_INT32_MAX				(PIL_DOUBLE)  2147483647.0
#define		PIL_UINT_MIN				(PIL_DOUBLE)  0
#define		PIL_UINT8_MAX				(PIL_DOUBLE)  255.0
#define		PIL_UINT16_MAX				(PIL_DOUBLE)  65535.0
#define		PIL_UINT32_MAX				(PIL_DOUBLE)  4294967295.0


/* pil_get_token() error codes */
/* --------------------------- */
#define		PIL_TOKEN_ERR_SYNTAX		(PIL_UINT16) 0
#define		PIL_TOKEN_ERR_BAD_NUM		(PIL_UINT16) 1
#define		PIL_TOKEN_ERR_NO_STREND		(PIL_UINT16) 2
#define		PIL_TOKEN_ERR_BAD_COMMENT	(PIL_UINT16) 3
#define		PIL_TOKEN_ERR_TOKEN_SIZE	(PIL_UINT16) 4


/* Some character definitions and EOF indicator value */
/* -------------------------------------------------- */
#define		PIL_END_OF_STRING		'\0'
#define		PIL_END_OF_LINE			'\n'
#define		PIL_REOF				(int) -1


/* Complexity level for polygon and closed path parsing */
/* ---------------------------------------------------- */
#define		PIL_SHAPE_SIMPLE		(PIL_UINT16) 0
#define		PIL_SHAPE_COMPLEX		(PIL_UINT16) 1


/* Destination for simple closed primitive parsing */
/* ----------------------------------------------- */
#define		PIL_CONTAINER_DEST		(PIL_UINT16) 0
#define		PIL_CLIPPER_DEST		(PIL_UINT16) 1


/* ---------------- */
/* Type definitions */
/* ---------------- */


/* Parser state information */
/* ------------------------ */
typedef struct
{
	PIL_UINT8			token [PIL_MAX_TOKEN_SIZE];
	PIL_UINT16			token_type;
	PIL_UINT16			token_error;
	PIL_DOUBLE			token_constant;
	PIL_BOOL			initial_fetch;
	int					char_code;
	PIL_UINT32			line_number;
	PIL_UINT16			state;
	PIL_UINT8			buf1 [PIL_MAX_TOKEN_SIZE];
	PIL_UINT8			buf2 [PIL_MAX_TOKEN_SIZE];
	PIL_UINT8			buf3 [PIL_MAX_TOKEN_SIZE];
	PIL_UINT8			buf4 [PIL_MAX_TOKEN_SIZE];
	PIL_UINT16			val1;
	PIL_UINT16			val2;
	PIL_UINT16			val3;
	PIL_UINT16			val4;
	PIL_CNT_ORDERING	cnt_ordering;
	PIL_CNT_ENCODING	cnt_encoding;
	PIL_UINT8			cnt_escape;
	PIL_UINT8			cnt_beg;
	PIL_UINT8			cnt_end;
	PIL_BOOL			cnt_start_state;
	PIL_BOOL			cnt_all_done;
}	PIL_PARSER_STATE;


/* Pointer to parser state information */
/* ----------------------------------- */
typedef PIL_PARSER_STATE PTRTYPE * PIL_PTR_PARSER_STATE;


/* ------------------------------------------------------------------------	*/
/*		P I L  G E N E R A T O R  S E C T I O N								*/
/* ------------------------------------------------------------------------	*/


/* ------------------------------------------ */
/* Platform specific strings, and preferences */
/* ------------------------------------------ */


/* --------------------------------------------------------------------------- *\
   Declare NEWLINE -- the string that the PIL generator uses to end each line 
   of a pil file.
   For Windows 3.0:
   #define NEWLINE ((char PTRTYPE*)("\r\n"))
   For Macintosh:
   #define NEWLINE	"\r"
   For most environments:
   #define NEWLINE	"\n"
\* --------------------------------------------------------------------------- */

#define NEWLINE		((char PTRTYPE*)("\n"))


/* ------------------------------------------------------ */
/* Separator to put between tokens (usually space or tab, */
/*	but could also be lf, nl, cr, ht, or any combination) */
/* ------------------------------------------------------ */

#define SEPARATOR	((char PTRTYPE*)(" "))


/* --------------------------------------------------------------------------- *\
	The PIL 4.0 Language Specification states that the begin block must be 
	a single { character and the end block must be a single } character. The 
	declarations below are provided in case PIL syntax is modified to support 
	multicharacter begin and end block strings.
\* --------------------------------------------------------------------------- */

#define BEGINBLOCK	((char PTRTYPE*)("{ "))
#define ENDBLOCK	((char PTRTYPE*)("}"))
#define BEGINLIST	((char PTRTYPE*)("\t\t\t{ "))


/* --------------------------------------------------------------------------- *\
   Declare PIL_BEGIN_COMMENT and PIL_END_COMMENT, the strings that introduce
   and end comments in a PIL file. Choose either the declaration for C style
   comments, or for C++ style comments. Note that for Windows, the appropiate
   pointer cast should be used.
\* --------------------------------------------------------------------------- */


#define	PIL_BEGIN_COMMENT	((char PTRTYPE*)("/* "))
#define PIL_END_COMMENT		((char PTRTYPE*)(" */"))
#if 0
#define	PIL_BEGIN_COMMENT	((char PTRTYPE*)("// "))
#define PIL_END_COMMENT		((char PTRTYPE*)(""))
#endif


/* -------------------- */
/* Keyword definitions	*/
/* -------------------- */


#define PIL_MIN_KW					1
#define	PIL_KW_OBJECT				1
#define	PIL_KW_TEXT_FLOW			2
#define	PIL_KW_GROUP				3
#define	PIL_KW_UNKNOWN				4
#define	PIL_KW_ASCII_TEXT			5
#define	PIL_KW_ATEX_ARIS			6
#define	PIL_KW_ATEX_ITF				7
#define	PIL_KW_EPS					8
#define	PIL_KW_ICL					9
#define	PIL_KW_IT8_1				10
#define	PIL_KW_IT8_2				11
#define	PIL_KW_MS_RTF				12
#define	PIL_KW_MS_WORD_3			13
#define	PIL_KW_MS_WORD_4			14
#define	PIL_KW_MIF					15
#define	PIL_KW_PICT					16
#define	PIL_KW_PIL_LAYOUT_NTET		17
#define	PIL_KW_POSTSCRIPT			18
#define	PIL_KW_QUARK_XPRESSTAGS		19
#define	PIL_KW_SCITEX_CT			20
#define	PIL_KW_SCITEX_HANDSHAKE		21
#define	PIL_KW_SCITEX_LW			22
#define	PIL_KW_SGML					23
#define	PIL_KW_TIFF					24
#define	PIL_KW_WP_100				25
#define	PIL_KW_WP_102				26
#define	PIL_KW_XYQUEST_XYWRITE		27
#define	PIL_KW_MAC_FILENAME			28
#define	PIL_KW_UNIX_FILENAME		29
#define	PIL_KW_MSDOS_FILENAME		30
#define	PIL_KW_OS2_FILENAME			31
#define	PIL_KW_VMS_FILENAME			32
#define	PIL_KW_VM_FILENAME			33
#define	PIL_KW_LCD_FILENAME			34
#define	PIL_KW_INLINE				35
#define	PIL_KW_NAME_TABLE			36
#define	PIL_KW_CANVAS				37
#define	PIL_KW_DIMENSIONS			38
#define	PIL_KW_UNITS				39
#define	PIL_KW_USER_NAME			40
#define	PIL_KW_TYPE					41
#define	PIL_KW_CLIPPER				42
#define	PIL_KW_APPLICATION			43
#define	PIL_KW_APP_NAME				44
#define	PIL_KW_APP_INT8				45
#define	PIL_KW_APP_INT16			46
#define	PIL_KW_APP_INT32			47
#define	PIL_KW_APP_UINT8			48
#define	PIL_KW_APP_UINT16			49
#define	PIL_KW_APP_UINT32			50
#define	PIL_KW_APP_DOUBLE			51
#define	PIL_KW_APP_STRING			52
#define	PIL_KW_FILL_RULE			53
#define	PIL_KW_RECTANGLE			54
#define	PIL_KW_CIRCLE				55
#define	PIL_KW_ELLIPSE				56
#define	PIL_KW_POLYGON				57
#define	PIL_KW_CLOSED_PATH			58
#define	PIL_KW_EVEN_ODD				59
#define	PIL_KW_NZ_WINDING			60
#define	PIL_KW_MOVE_TO				61
#define	PIL_KW_LINE_TO				62
#define	PIL_KW_CURVE_TO				63
#define	PIL_KW_ARC_TO				64
#define	PIL_KW_FLOW_LABEL			65
#define	PIL_KW_FLOW_OBJECTS			66
#define	PIL_KW_FLOW_FROM			67
#define	PIL_KW_FLOW_TO				68
#define	PIL_KW_FLOW_CONTENT			69
#define	PIL_KW_ID					70
#define	PIL_KW_GROUP_OBJECTS		71
#define	PIL_KW_ORIGIN				72
#define	PIL_KW_ROT_ANGLE			73
#define	PIL_KW_ROT_POINT			74
#define	PIL_KW_CONTAINER			75
#define	PIL_KW_GRAPHIC				76
#define	PIL_KW_RC_TYPE				77
#define	PIL_KW_RC_NAME				78
#define	PIL_KW_BBOX					79
#define	PIL_KW_RC_TYPE_TEXT			80
#define	PIL_KW_RC_TYPE_GEOMETRY		81
#define	PIL_KW_RC_TYPE_LINEART		82
#define	PIL_KW_RC_TYPE_IMAGE		83
#define	PIL_KW_COLOR_MODEL			84
#define	PIL_KW_DEV_GRAY				85
#define	PIL_KW_CAL_GRAY				86
#define	PIL_KW_DEV_RGB				87
#define	PIL_KW_CAL_RGB				88
#define	PIL_KW_DEV_CMYK				89
#define	PIL_KW_SPOT					90
#define	PIL_KW_PANTONE				91
#define	PIL_KW_RENDER_OP			92
#define	PIL_KW_STROKE				93
#define	PIL_KW_FILL					94
#define	PIL_KW_FILL_STROKE			95
#define	PIL_KW_RENDER_ATTR			96
#define	PIL_KW_STROKE_COLOR			97
#define	PIL_KW_POSITION				98
#define	PIL_KW_WIDTH				99
#define	PIL_KW_CAP					100
#define	PIL_KW_JOIN					101
#define	PIL_KW_MITER_LIMIT			102
#define	PIL_KW_FILL_COLOR			103
#define	PIL_KW_INSIDE				104
#define	PIL_KW_OUTSIDE				105
#define	PIL_KW_CENTERED				106
#define	PIL_KW_BUTT_CAP				107
#define	PIL_KW_ROUND_CAP			108
#define	PIL_KW_PROJECTING_CAP		109
#define	PIL_KW_MITER_JOIN			110
#define	PIL_KW_BEVEL_JOIN			111
#define	PIL_KW_ROUND_JOIN			112
#define	PIL_KW_LINE					113
#define	PIL_KW_ARC					114
#define	PIL_KW_BEZIER				115
#define	PIL_KW_POLY_LINE			116
#define	PIL_KW_OPEN_PATH			117
#define	PIL_KW_DATA					118
#define	PIL_KW_HEADER				119
#define	PIL_KW_LITTLE_ENDIAN		120
#define	PIL_KW_BIG_ENDIAN			121
#define	PIL_KW_ASCII				122
#define	PIL_KW_BINARY				123
#define PIL_KW_LAYOUT				124
#define PIL_KW_CONTENT				125
#define	PIL_KW_LAYOUT_VS1			126
#define	PIL_KW_LAYOUT_VS2			127
#define	PIL_KW_LAYOUT_VS3			128
#define	PIL_KW_LAYOUT_VS4			129
#define	PIL_KW_CONTENT_VS1			130
#define	PIL_KW_CONTENT_VS2			131
#define	PIL_KW_CONTENT_VS3			132
#define	PIL_KW_CONTENT_VS4			133
#define PIL_KW_CLIPPER_SHAPE		134
#define PIL_KW_CONTAINER_SHAPE		135
#define PIL_KW_OBJ_TEXT				136
#define PIL_KW_OBJ_PICTURE			137
#define PIL_KW_OBJ_PRIMITIVE		138		
#define	PIL_SIZE_KW_ARRAY			139


/* ------------------------------------------------------------------------	*/
/*		P I L  F U N C T I O N  P R O T O T Y P E S							*/
/* ------------------------------------------------------------------------	*/


#ifdef		PIL_ANSI_PROTOTYPES
#define		PIL_VOID				void

	
	/* ------------------------ */
	/* API Mainstream functions */
	/* ------------------------ */
	
	
	PIL_ERROR pil_pg_init					
	(
		char PTRTYPE *		bufptr, 
		PIL_UINT32 			buflen, 
		int 		 		(FPTRTYPE *fp_getc)(PIL_VOID),
		PIL_VOID	 		(FPTRTYPE *fp_putc)(PIL_UINT8),
		PIL_UINT32 			cntbsize,
		PIL_INT32 			asi_count,
		char PTRPTR 		asi_names
	);
	PIL_ERROR pil_create_component		
	(
		pil_component PTRPTR	comp, 
		PIL_COMP_TYPE 			comp_type
	);
	PIL_ERROR pil_get_component	(pil_component 	PTRPTR 		comp);
	PIL_ERROR pil_put_component	(pil_component 	PTRTYPE * 	comp);
	PIL_VOID pil_put_comment	(char 			PTRTYPE * 	comment);
	
	
	/* ------------------------------------------------------ */
	/* API Functions for controlling parser/generator actions */
	/* ------------------------------------------------------ */


	PIL_ERROR pil_set_asa_list			
	(
		PIL_INT32 		asi_count, 
		char PTRPTR 	asi_names
	);
	PIL_ERROR pil_set_buffer				
	(
		char PTRTYPE 	*bufptr, 
		PIL_UINT32 		buflen
	);
	PIL_ERROR pil_set_getc		(int 		 	(FPTRTYPE *fp_getc)(PIL_VOID));
	PIL_ERROR pil_set_putc		(PIL_VOID 		(FPTRTYPE *fp_putc)(PIL_UINT8));
	PIL_ERROR pil_set_cnt_bsize	(PIL_UINT32 	cntbsize);
	
	
	/* -------------------- */
	/* API Status functions */
	/* -------------------- */
	
	
	PIL_ERROR pil_get_version				
	(
		char PTRPTR, 
		char PTRPTR
	);
	PIL_ERROR 	pil_get_mem_avail 	(PIL_UINT32 PTRTYPE *);
	PIL_ERROR 	pil_last_error		(PIL_VOID);
	PIL_UINT32	pil_get_line_num	(PIL_VOID);


	/* ------------------------------------- */
	/* API Data structure creation functions */
	/* ------------------------------------- */
	
	
	PIL_ERROR pil_alloc					
	(
		PIL_UINT32, 
		PIL_VOID PTRPTR
	);
	PIL_ERROR pil_create_asa				
	(
		pil_asa PTRPTR, 
		char PTRTYPE *,
		PIL_UINT32, 
		PIL_ASA_VAL_TYPE,
		char PTRTYPE *
	);
	PIL_ERROR pil_create_asi				
	(
		pil_asi PTRPTR, 
		char PTRTYPE *
	);
	PIL_ERROR pil_create_nametbleentry	
	(
		pil_name_table_entry PTRPTR, 
		char PTRTYPE *,
		PIL_NT_TYPE, 
		char PTRTYPE *, 
		PIL_NT_DMN, 
		char PTRTYPE *, 
		PIL_NT_VAL, 
		char PTRTYPE *
	);
	PIL_ERROR pil_create_objid			
	(
		pil_object_id_list PTRPTR,
		char PTRTYPE *
	);
	PIL_ERROR pil_create_polypt			
	(
		pil_poly_pt PTRPTR,
		PIL_INT32, 
		PIL_INT32
	);
	PIL_ERROR pil_create_pathpt			
	(
		pil_path_pt PTRPTR,
		PIL_PATH_PART,
		PIL_INT32,
		PIL_INT32,
		PIL_INT32,
		PIL_INT32,
		PIL_INT32,
		PIL_INT32,
		PIL_INT32
	);
	PIL_ERROR pil_create_pvalue			
	(
		pil_value PTRPTR,
		PIL_ASA_VAL_TYPE,
		FAST char PTRTYPE *
	);
	PIL_ERROR pil_create_canvas			(pil_canvas 		PTRPTR);
	PIL_ERROR pil_create_content_end	(pil_content_end 	PTRPTR);
	PIL_ERROR pil_create_content_hdr	(pil_content_hdr 	PTRPTR);
	PIL_ERROR pil_create_content_data	(pil_content_data 	PTRPTR);
	PIL_ERROR pil_create_content_start	(pil_content_start 	PTRPTR);
	PIL_ERROR pil_create_group			(pil_group 			PTRPTR);
	PIL_ERROR pil_create_layout_end		(pil_layout_end 	PTRPTR);
	PIL_ERROR pil_create_layout_start	(pil_layout_start 	PTRPTR);
	PIL_ERROR pil_create_name_table		(pil_name_table 	PTRPTR);
	PIL_ERROR pil_create_object			(pil_object 		PTRPTR);
	PIL_ERROR pil_create_text_flow		(pil_text_flow 		PTRPTR);
	
	
	/* ------------------------------ */
	/* API Internal utility functions */
	/* ------------------------------ */
	
	
	PIL_VOID pil_clear					
	(
		FAST char PTRTYPE *,
		FAST PIL_UINT32
	);
	PIL_ERROR pil_append_string			
	(
		char PTRTYPE *, 
		char PTRPTR
	);
	PIL_INT16 pil_strcmp					
	(
		char PTRTYPE *str1, 
		char PTRTYPE *str2
	);
	char PTRTYPE * pil_strcpy		
	(
		char PTRTYPE *dest, 
		char PTRTYPE *src
	);
	PIL_DOUBLE	pil_atof		(char PTRTYPE *ascii_num);
	PIL_UINT32	pil_strlen		(char PTRTYPE *string);
	PIL_ERROR 	pil_set_error 	(PIL_ERROR);


	/* ------------------------------------------------ */
	/* Parser prototypes								*/
	/* ------------------------------------------------ */
	
	
	/* ------------------------ */
	/* Initialization functions */
	/* ------------------------ */
	
	PIL_VOID 	pil_init_parser 			(PIL_VOID);
	PIL_VOID 	pil_init_char_class_table 	(PIL_VOID);
	PIL_VOID 	pil_init_parser_state 		(PIL_VOID);
	PIL_VOID	pil_init_keywords			(PIL_VOID);
	
	
	/* ------------------- */
	/* Tokenizer functions */
	/* ------------------- */
	
	PIL_UINT16	pil_get_token 				(PIL_VOID);
	PIL_BOOL	pil_is_token_int8 			(PIL_VOID);
	PIL_BOOL	pil_is_token_int16 			(PIL_VOID);
	PIL_BOOL	pil_is_token_int32 			(PIL_VOID);
	PIL_BOOL 	pil_is_token_uint8 			(PIL_VOID);
	PIL_BOOL 	pil_is_token_uint16 		(PIL_VOID);
	PIL_BOOL 	pil_is_token_uint32 		(PIL_VOID);
	PIL_BOOL 	pil_is_token_double 		(PIL_VOID);
	PIL_BOOL 	pil_is_token_string 		(PIL_VOID);
	PIL_BOOL 	pil_is_token_symbol 		(PIL_VOID);
	PIL_BOOL 	pil_is_token_beg_block 		(PIL_VOID);
	PIL_BOOL 	pil_is_token_end_block 		(PIL_VOID);
	PIL_BOOL 	pil_is_token_error 			(PIL_VOID);
	PIL_BOOL 	pil_is_token_eof 			(PIL_VOID);
	
	
	/* ------------------------ */
	/* Keyword tester functions */
	/* ------------------------ */
	
	PIL_BOOL	pil_is_kw_pil_layout		(PIL_VOID);
	PIL_BOOL	pil_is_kw_pil_content		(PIL_VOID);
	PIL_BOOL	pil_is_kw_object			(PIL_VOID);
	PIL_BOOL	pil_is_kw_text_flow			(PIL_VOID);
	PIL_BOOL	pil_is_kw_group				(PIL_VOID);
	PIL_BOOL 	pil_is_kw_unknown 			(PIL_VOID);
	PIL_BOOL 	pil_is_kw_ascii_text 		(PIL_VOID);
	PIL_BOOL 	pil_is_kw_atex_aris 		(PIL_VOID);
	PIL_BOOL 	pil_is_kw_atex_itf 			(PIL_VOID);
	PIL_BOOL 	pil_is_kw_eps 				(PIL_VOID);
	PIL_BOOL 	pil_is_kw_icl 				(PIL_VOID);
	PIL_BOOL 	pil_is_kw_it8_1 			(PIL_VOID);
	PIL_BOOL 	pil_is_kw_it8_2 			(PIL_VOID);
	PIL_BOOL 	pil_is_kw_ms_rtf 			(PIL_VOID);				
	PIL_BOOL 	pil_is_kw_ms_word_3 		(PIL_VOID);			
	PIL_BOOL 	pil_is_kw_ms_word_4 		(PIL_VOID);			
	PIL_BOOL 	pil_is_kw_mif 				(PIL_VOID);					
	PIL_BOOL 	pil_is_kw_pict 				(PIL_VOID);				
	PIL_BOOL 	pil_is_kw_pil_layout_ntet 	(PIL_VOID);			
	PIL_BOOL 	pil_is_kw_postscript 		(PIL_VOID);			
	PIL_BOOL 	pil_is_kw_quark_xpresstags 	(PIL_VOID);	
	PIL_BOOL 	pil_is_kw_scitex_ct 		(PIL_VOID);			
	PIL_BOOL 	pil_is_kw_scitex_handshake 	(PIL_VOID);	
	PIL_BOOL 	pil_is_kw_scitex_lw 		(PIL_VOID);			
	PIL_BOOL 	pil_is_kw_sgml 				(PIL_VOID);				
	PIL_BOOL 	pil_is_kw_tiff 				(PIL_VOID);				
	PIL_BOOL 	pil_is_kw_wp_100 			(PIL_VOID);				
	PIL_BOOL 	pil_is_kw_wp_102 			(PIL_VOID);				
	PIL_BOOL 	pil_is_kw_xyquest_xywrite 	(PIL_VOID);		
	PIL_BOOL 	pil_is_kw_mac_filename 		(PIL_VOID);		
	PIL_BOOL 	pil_is_kw_unix_filename 	(PIL_VOID);		
	PIL_BOOL 	pil_is_kw_msdos_filename 	(PIL_VOID);		
	PIL_BOOL 	pil_is_kw_os2_filename 		(PIL_VOID);		
	PIL_BOOL 	pil_is_kw_vms_filename 		(PIL_VOID);		
	PIL_BOOL 	pil_is_kw_vm_filename 		(PIL_VOID);			
	PIL_BOOL 	pil_is_kw_lcd_filename 		(PIL_VOID);		
	PIL_BOOL 	pil_is_kw_inline 			(PIL_VOID);				
	PIL_BOOL 	pil_is_kw_name_table 		(PIL_VOID);		
	PIL_BOOL	pil_is_kw_nt_type			(PIL_VOID);			
	PIL_BOOL	pil_is_kw_nt_domain			(PIL_VOID);
	PIL_BOOL	pil_is_kw_nt_value			(PIL_VOID);
	PIL_BOOL	pil_is_kw_canvas			(PIL_VOID);
	PIL_BOOL	pil_is_kw_dimensions		(PIL_VOID);
	PIL_BOOL	pil_is_kw_units				(PIL_VOID);
	PIL_BOOL	pil_is_kw_user_name			(PIL_VOID);
	PIL_BOOL	pil_is_kw_type				(PIL_VOID);
	PIL_BOOL	pil_is_kw_clipper			(PIL_VOID);
	PIL_BOOL	pil_is_kw_application		(PIL_VOID);
	PIL_BOOL	pil_is_kw_app_name			(PIL_VOID);
	PIL_BOOL	pil_is_kw_app_type			(PIL_VOID);
	PIL_BOOL	pil_is_kw_app_int8			(PIL_VOID);
	PIL_BOOL	pil_is_kw_app_int16			(PIL_VOID);
	PIL_BOOL	pil_is_kw_app_int32			(PIL_VOID);
	PIL_BOOL	pil_is_kw_app_uint8			(PIL_VOID);
	PIL_BOOL	pil_is_kw_app_uint16		(PIL_VOID);
	PIL_BOOL	pil_is_kw_app_uint32		(PIL_VOID);
	PIL_BOOL	pil_is_kw_app_double		(PIL_VOID);
	PIL_BOOL	pil_is_kw_app_string		(PIL_VOID);
	PIL_BOOL	pil_is_kw_fill_rule			(PIL_VOID);
	PIL_BOOL	pil_is_kw_rectangle			(PIL_VOID);
	PIL_BOOL	pil_is_kw_circle			(PIL_VOID);
	PIL_BOOL	pil_is_kw_ellipse			(PIL_VOID);
	PIL_BOOL	pil_is_kw_polygon			(PIL_VOID);
	PIL_BOOL	pil_is_kw_closed_path		(PIL_VOID);
	PIL_BOOL	pil_is_kw_even_odd			(PIL_VOID);
	PIL_BOOL	pil_is_kw_nz_winding		(PIL_VOID);
	PIL_BOOL	pil_is_kw_move_to			(PIL_VOID);
	PIL_BOOL	pil_is_kw_line_to			(PIL_VOID);
	PIL_BOOL	pil_is_kw_curve_to			(PIL_VOID);
	PIL_BOOL	pil_is_kw_arc_to			(PIL_VOID);
	PIL_BOOL	pil_is_kw_flow_label		(PIL_VOID);
	PIL_BOOL	pil_is_kw_flow_objects		(PIL_VOID);
	PIL_BOOL	pil_is_kw_flow_from			(PIL_VOID);
	PIL_BOOL	pil_is_kw_flow_to			(PIL_VOID);
	PIL_BOOL	pil_is_kw_flow_content		(PIL_VOID);
	PIL_BOOL	pil_is_kw_id				(PIL_VOID);
	PIL_BOOL	pil_is_kw_group_objects		(PIL_VOID);
	PIL_BOOL	pil_is_kw_origin			(PIL_VOID);
	PIL_BOOL	pil_is_kw_rot_angle			(PIL_VOID);
	PIL_BOOL	pil_is_kw_rot_point			(PIL_VOID);
	PIL_BOOL	pil_is_kw_container			(PIL_VOID);
	PIL_BOOL	pil_is_kw_graphic			(PIL_VOID);
	PIL_BOOL	pil_is_kw_rc_type			(PIL_VOID);
	PIL_BOOL	pil_is_kw_rc_name			(PIL_VOID);
	PIL_BOOL	pil_is_kw_bbox				(PIL_VOID);
	PIL_BOOL	pil_is_kw_rc_type_text		(PIL_VOID);
	PIL_BOOL	pil_is_kw_rc_type_geometry	(PIL_VOID);
	PIL_BOOL	pil_is_kw_rc_type_lineart	(PIL_VOID);
	PIL_BOOL	pil_is_kw_rc_type_image		(PIL_VOID);
	PIL_BOOL	pil_is_kw_color_model		(PIL_VOID);
	PIL_BOOL	pil_is_kw_dev_gray			(PIL_VOID);
	PIL_BOOL	pil_is_kw_cal_gray			(PIL_VOID);
	PIL_BOOL	pil_is_kw_dev_rgb			(PIL_VOID);
	PIL_BOOL	pil_is_kw_cal_rgb			(PIL_VOID);
	PIL_BOOL	pil_is_kw_dev_cmyk			(PIL_VOID);
	PIL_BOOL	pil_is_kw_spot				(PIL_VOID);
	PIL_BOOL	pil_is_kw_pantone			(PIL_VOID);
	PIL_BOOL	pil_is_kw_render_op			(PIL_VOID);
	PIL_BOOL	pil_is_kw_stroke			(PIL_VOID);
	PIL_BOOL	pil_is_kw_fill				(PIL_VOID);
	PIL_BOOL	pil_is_kw_fill_stroke		(PIL_VOID);
	PIL_BOOL	pil_is_kw_render_attr		(PIL_VOID);
	PIL_BOOL	pil_is_kw_stroke_color		(PIL_VOID);
	PIL_BOOL	pil_is_kw_position			(PIL_VOID);
	PIL_BOOL	pil_is_kw_width				(PIL_VOID);
	PIL_BOOL	pil_is_kw_cap				(PIL_VOID);
	PIL_BOOL	pil_is_kw_join				(PIL_VOID);
	PIL_BOOL	pil_is_kw_miter_limit		(PIL_VOID);
	PIL_BOOL	pil_is_kw_fill_color		(PIL_VOID);
	PIL_BOOL	pil_is_kw_inside			(PIL_VOID);
	PIL_BOOL	pil_is_kw_outside			(PIL_VOID);
	PIL_BOOL	pil_is_kw_centered			(PIL_VOID);
	PIL_BOOL	pil_is_kw_butt_cap			(PIL_VOID);
	PIL_BOOL	pil_is_kw_round_cap			(PIL_VOID);
	PIL_BOOL	pil_is_kw_projecting_cap	(PIL_VOID);
	PIL_BOOL	pil_is_kw_miter_join		(PIL_VOID);
	PIL_BOOL	pil_is_kw_bevel_join		(PIL_VOID);
	PIL_BOOL	pil_is_kw_round_join		(PIL_VOID);
	PIL_BOOL	pil_is_kw_line				(PIL_VOID);
	PIL_BOOL	pil_is_kw_arc				(PIL_VOID);
	PIL_BOOL	pil_is_kw_bezier			(PIL_VOID);
	PIL_BOOL	pil_is_kw_poly_line			(PIL_VOID);
	PIL_BOOL	pil_is_kw_open_path			(PIL_VOID);
	PIL_BOOL	pil_is_kw_cm_value			(PIL_VOID);
	PIL_BOOL	pil_is_kw_rop_value			(PIL_VOID);
	PIL_BOOL	pil_is_kw_data				(PIL_VOID);
	PIL_BOOL	pil_is_kw_header			(PIL_VOID);
	PIL_BOOL	pil_is_kw_little_endian		(PIL_VOID);
	PIL_BOOL	pil_is_kw_big_endian		(PIL_VOID);
	PIL_BOOL	pil_is_kw_ascii				(PIL_VOID);
	PIL_BOOL	pil_is_kw_binary			(PIL_VOID);
	PIL_BOOL 	pil_is_kw_text				(PIL_VOID);
	PIL_BOOL 	pil_is_kw_picture			(PIL_VOID);
	PIL_BOOL 	pil_is_kw_primitive			(PIL_VOID);
	PIL_BOOL 	pil_is_kw_clipper_shape		(PIL_VOID);
	PIL_BOOL 	pil_is_kw_container_shape	(PIL_VOID);
	
	
	/* -------------------------------------------- */
	/* Component and sub-component parser functions */
	/* -------------------------------------------- */
	
	PIL_ERROR pil_parse_nametable 		(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_canvas 			(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_object 			(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_text_flow 		(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_group 			(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_layout_start	(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_dimensions 		(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_units			(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_user_name 		(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_type 			(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_asi		 		(pil_asi PTRPTR asi);
	PIL_ERROR pil_parse_app_attr		(pil_asa PTRPTR asa);
	PIL_ERROR pil_parse_rectangle		(pil_rectangle PTRTYPE *rectangle);
	PIL_ERROR pil_parse_circle			(pil_circle PTRTYPE *circle);
	PIL_ERROR pil_parse_ellipse			(pil_ellipse PTRTYPE *ellipse);
	PIL_ERROR pil_parse_move_to			(pil_point PTRTYPE *p);
	PIL_ERROR pil_parse_line_to			(pil_point PTRTYPE *p);
	PIL_ERROR pil_parse_point			(pil_point PTRTYPE *p);
	PIL_ERROR pil_parse_line			(pil_line PTRTYPE *line);
	PIL_ERROR pil_parse_arc				(pil_arc PTRTYPE *arc);
	PIL_ERROR pil_parse_bezier			(pil_bezier PTRTYPE *bezier);
	PIL_ERROR pil_parse_poly_line		(pil_polyline PTRTYPE *poly_line);
	PIL_ERROR pil_parse_open_path		(pil_open_path PTRTYPE *open_path);
	PIL_ERROR pil_parse_stroke_position	(pil_graphic PTRTYPE *graphic);
	PIL_ERROR pil_parse_stroke_width	(pil_graphic PTRTYPE *graphic);
	PIL_ERROR pil_parse_stroke_cap		(pil_graphic PTRTYPE *graphic);
	PIL_ERROR pil_parse_stroke_join		(pil_graphic PTRTYPE *graphic);
	PIL_ERROR pil_parse_stroke_mlimit	(pil_graphic PTRTYPE *graphic);
	PIL_ERROR pil_parse_fill_rule		(pil_graphic PTRTYPE *graphic);
	PIL_ERROR pil_parse_flow_label		(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_flow_objects	(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_flow_from		(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_flow_to			(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_flow_content	(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_group_id		(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_group_objects	(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_origin 			(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_id 				(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_rot_angle 		(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_rot_point 		(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_graphic 		(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_rc_type 		(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_rc_name 		(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_bbox 			(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_cnt_start 		(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_cnt_header 		(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_ascii_cnt 		(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_binary_cnt 		(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_parse_simp_closed_prim 	
	(
		pil_component PTRPTR pil_comp, 
		PIL_UINT16 dest
	);
	PIL_ERROR pil_parse_polygon			
	(
		pil_polygon PTRTYPE *polygon, 
		PIL_UINT16 clevel
	);
	PIL_ERROR pil_parse_closed_path		
	(
		pil_closed_path PTRTYPE *path, 
		PIL_UINT16 clevel
	);
	PIL_ERROR pil_parse_curve_to			
	(
		pil_point PTRTYPE *p1, 
		pil_point PTRTYPE *p2, 
		pil_point PTRTYPE *p3
	);
	PIL_ERROR pil_parse_arc_to			
	(
		pil_point PTRTYPE *p1, 
		pil_point PTRTYPE *p2, 
		PIL_INT32 PTRTYPE *r
	);
	PIL_ERROR pil_parse_color_value		
	(
		pil_color PTRTYPE *color_value, 
		PIL_COLOR_MODEL color_model
	);
	
	
	/* --------------------------------------- */
	/* Parser functions for structure creation */
	/* --------------------------------------- */
	
	PIL_ERROR pil_make_layout_start_comp 	(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_make_layout_end_comp 		(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_make_name_table_comp 		(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_make_name_table_entry 	(pil_name_table_entry PTRPTR entry);
	PIL_ERROR pil_make_canvas_comp 			(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_make_text_flow_comp		(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_make_group_comp			(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_make_object_comp 			(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_make_cnt_start_comp 		(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_make_cnt_end_comp 		(pil_component PTRPTR pil_comp);
	PIL_ERROR pil_make_cnt_header_comp		(pil_component PTRPTR pil_comp);
	
	
	/* ------------------------ */
	/* Parser utility functions */
	/* ------------------------ */
	
	PIL_VOID	pil_copy_to_buf1		(PIL_VOID);
	PIL_VOID	pil_copy_to_buf2		(PIL_VOID);
	PIL_VOID	pil_copy_to_buf3		(PIL_VOID);
	PIL_VOID	pil_copy_to_buf4		(PIL_VOID);
	PIL_ERROR 	pil_spinner				(PIL_VOID);
	PIL_BOOL	pil_str_eq				(char PTRTYPE *string);
	PIL_UINT8	pil_combine_hex_digits	(FAST PIL_UINT8 hex1, FAST PIL_UINT8 hex2);
	PIL_BOOL	pil_is_pil_file			(PIL_VOID);
	PIL_BOOL	pil_is_desired_asi		(PIL_VOID);
	PIL_BOOL	pil_is_hex_digit		(FAST PIL_UINT8 char_code);
	PIL_BOOL	pil_is_white_space		(FAST PIL_UINT8 char_code);
	PIL_BOOL	pil_is_graphic_prim		(PIL_VOID);


	/* ------------------------------------------------ */
	/* Generator prototypes								*/
	/* ------------------------------------------------ */

	
	char PTRTYPE *pil_token_to_str			(int);
	char PTRTYPE *pil_gen_indents			(int);
	PIL_ERROR 	pil_put_arc					(pil_arc PTRTYPE *);
	PIL_ERROR 	pil_put_asa					(pil_asa PTRTYPE *);
	PIL_ERROR 	pil_put_asi					(pil_asi PTRTYPE *);
	PIL_ERROR 	pil_put_bezier				(pil_bezier PTRTYPE *);
	PIL_ERROR 	pil_put_canvas				(pil_canvas PTRTYPE *);
	PIL_ERROR 	pil_put_circledata			(pil_circle PTRTYPE *);
	PIL_ERROR 	pil_put_clipper				(pil_clipper PTRTYPE *);
	PIL_ERROR 	pil_put_closed_shape		(pil_closed_primitive PTRTYPE *);
	PIL_ERROR 	pil_put_color				(PIL_UINT16,pil_color PTRTYPE *);
	PIL_ERROR 	pil_put_container			(pil_container PTRTYPE *);
	PIL_ERROR 	pil_put_content_data		(pil_content_data PTRTYPE *);
	PIL_ERROR 	pil_put_content_end			(pil_content_end PTRTYPE *);
	PIL_ERROR 	pil_put_content_hdr			(pil_content_hdr PTRTYPE *);
	PIL_ERROR 	pil_put_content_start		(pil_content_start PTRTYPE *);
	PIL_ERROR 	pil_put_ctype_code			(PIL_NT_TYPE, char PTRTYPE *);
	PIL_ERROR 	pil_put_data_code			(PIL_UINT16);
	PIL_ERROR 	pil_put_dimensions			(PIL_UINT32, PIL_UINT32);
	PIL_ERROR 	pil_put_domain_code			(PIL_NT_DMN, char PTRTYPE *);
	PIL_ERROR 	pil_put_graphic				(pil_graphic PTRTYPE *);
	PIL_ERROR 	pil_put_group				(pil_group PTRTYPE *);
	PIL_ERROR 	pil_put_layout_end			(pil_layout_end PTRTYPE *);
	PIL_ERROR 	pil_put_layout_start		(pil_layout_start PTRTYPE *);
	PIL_ERROR 	pil_put_line				(pil_line PTRTYPE *);
	PIL_ERROR 	pil_put_nametbleentry		(pil_name_table_entry PTRTYPE *);
	PIL_ERROR 	pil_put_name_table			(pil_name_table PTRTYPE *);
	PIL_ERROR 	pil_put_object				(pil_object PTRTYPE *);
	PIL_ERROR 	pil_put_objid				(pil_object_id_list PTRTYPE *);
	PIL_ERROR 	pil_put_pathdata			(pil_path_pt PTRTYPE *);
	PIL_ERROR 	pil_put_polydata			(pil_poly_pt PTRTYPE *);
	PIL_ERROR 	pil_put_pvalue				(pil_value PTRTYPE *, PIL_ASA_VAL_TYPE);
	PIL_ERROR 	pil_put_rectdata			(pil_rectangle PTRTYPE *);
	PIL_ERROR 	pil_put_shape				(pil_primitive PTRTYPE *);
	PIL_ERROR 	pil_put_shape_code			(PIL_UINT16);
	PIL_ERROR 	pil_put_text_flow			(pil_text_flow PTRTYPE *);
	PIL_VOID	pil_ftoa					(PIL_DOUBLE, char PTRTYPE *);
	PIL_VOID	pil_i32toa					(FAST PIL_INT32, FAST char PTRTYPE *);
	PIL_VOID	pil_nextline				(PIL_VOID);	
	PIL_VOID	pil_put_dbl					(PIL_DOUBLE);
	PIL_VOID	pil_put_hex					(PIL_UINT8);
	PIL_VOID	pil_put_i32					(PIL_INT32);
	PIL_VOID	pil_put_kw					(int);
	PIL_VOID	pil_put_kw_no_sep			(int);
	PIL_VOID	pil_put_literal				(char PTRTYPE *);
	PIL_VOID	pil_put_named_begblk		(int);
	PIL_VOID	pil_put_named_dbl			(int, PIL_DOUBLE);
	PIL_VOID	pil_put_named_i32			(int, PIL_INT32);
	PIL_VOID	pil_put_named_str			(int, char PTRTYPE *);
	PIL_VOID	pil_put_named_ui32			(int, PIL_UINT32);
	PIL_VOID	pil_put_nested_begblk		(PIL_VOID);
	PIL_VOID	pil_put_nested_endblk		(PIL_VOID);
	PIL_VOID	pil_put_point				(pil_point PTRTYPE *);
	PIL_VOID	pil_put_point_data			(pil_point PTRTYPE *);
	PIL_VOID	pil_put_quoted_str			(char PTRTYPE *);
	PIL_VOID	pil_put_string				(char PTRTYPE *);
	PIL_VOID	pil_put_ui32				(PIL_UINT32);
	PIL_VOID	pil_to_hex					(PIL_UINT8, char PTRTYPE *);
	PIL_VOID	pil_ui32toa					(FAST PIL_UINT32, FAST char PTRTYPE *);
	PIL_VOID	pil_set_dflt_hdr			(PIL_VOID);


#else

/* ------------------------- */
/* K&R Function declarations */
/* ------------------------- */
#define		PIL_VOID	int


	/* ------------------------ */
	/* API Mainstream functions */
	/* ------------------------ */
	
	
	PIL_ERROR pil_pg_init			();					
	PIL_ERROR pil_create_component	();		
	PIL_ERROR pil_get_component		();
	PIL_ERROR pil_put_component		();
	PIL_VOID  pil_put_comment		();
	
	
	/* ------------------------------------------------------ */
	/* API Functions for controlling parser/generator actions */
	/* ------------------------------------------------------ */


	PIL_ERROR pil_set_asa_list	();			
	PIL_ERROR pil_set_buffer	();				
	PIL_ERROR pil_set_getc		();
	PIL_ERROR pil_set_putc		();
	PIL_ERROR pil_set_cnt_bsize	();
	
	
	/* -------------------- */
	/* API Status functions */
	/* -------------------- */
	
	
	PIL_ERROR pil_get_version		();				
	PIL_ERROR pil_get_mem_avail 	();
	PIL_ERROR pil_last_error		();
	PIL_UINT32	pil_get_line_num	();


	/* ------------------------------------- */
	/* API Data structure creation functions */
	/* ------------------------------------- */
	
	
	PIL_ERROR pil_alloc					();					
	PIL_ERROR pil_create_asa			();				
	PIL_ERROR pil_create_asi			();				
	PIL_ERROR pil_create_nametbleentry	();	
	PIL_ERROR pil_create_objid			();		
	PIL_ERROR pil_create_polypt			();		
	PIL_ERROR pil_create_pathpt			();		
	PIL_ERROR pil_create_pvalue			();			
	PIL_ERROR pil_create_canvas			();
	PIL_ERROR pil_create_content_end	();
	PIL_ERROR pil_create_content_hdr	();
	PIL_ERROR pil_create_content_data	();
	PIL_ERROR pil_create_content_start	();
	PIL_ERROR pil_create_group			();
	PIL_ERROR pil_create_layout_end		();
	PIL_ERROR pil_create_layout_start	();
	PIL_ERROR pil_create_name_table		();
	PIL_ERROR pil_create_object			();
	PIL_ERROR pil_create_text_flow		();
	
	
	/* ------------------------------ */
	/* API Internal utility functions */
	/* ------------------------------ */
	
	
	PIL_VOID 		pil_clear			();					
	PIL_ERROR 		pil_append_string	();			
	PIL_INT16		pil_strcmp			();					
	char PTRTYPE * 	pil_strcpy			();		
	PIL_DOUBLE		pil_atof			();
	PIL_UINT32		pil_strlen			();
	PIL_ERROR 		pil_set_error 		();


	/* ------------------------------------------------ */
	/* Parser prototypes								*/
	/* ------------------------------------------------ */
	
	
	/* ------------------------ */
	/* Initialization functions */
	/* ------------------------ */
	
	PIL_VOID 	pil_init_parser 			();
	PIL_VOID 	pil_init_char_class_table 	();
	PIL_VOID 	pil_init_parser_state 		();
	PIL_VOID	pil_init_keywords			();
	
	
	/* ------------------- */
	/* Tokenizer functions */
	/* ------------------- */
	
	PIL_UINT16	pil_get_token 				();
	PIL_BOOL	pil_is_token_int8 			();
	PIL_BOOL	pil_is_token_int16 			();
	PIL_BOOL	pil_is_token_int32 			();
	PIL_BOOL 	pil_is_token_uint8 			();
	PIL_BOOL 	pil_is_token_uint16 		();
	PIL_BOOL 	pil_is_token_uint32 		();
	PIL_BOOL 	pil_is_token_double 		();
	PIL_BOOL 	pil_is_token_string 		();
	PIL_BOOL 	pil_is_token_symbol 		();
	PIL_BOOL 	pil_is_token_beg_block 		();
	PIL_BOOL 	pil_is_token_end_block 		();
	PIL_BOOL 	pil_is_token_error 			();
	PIL_BOOL 	pil_is_token_eof 			();
	
	
	/* ------------------------ */
	/* Keyword tester functions */
	/* ------------------------ */
	
	PIL_BOOL	pil_is_kw_pil_layout		();
	PIL_BOOL	pil_is_kw_pil_content		();
	PIL_BOOL	pil_is_kw_object			();
	PIL_BOOL	pil_is_kw_text_flow			();
	PIL_BOOL	pil_is_kw_group				();
	PIL_BOOL 	pil_is_kw_unknown 			();
	PIL_BOOL 	pil_is_kw_ascii_text 		();
	PIL_BOOL 	pil_is_kw_atex_aris 		();
	PIL_BOOL 	pil_is_kw_atex_itf 			();
	PIL_BOOL 	pil_is_kw_eps 				();
	PIL_BOOL 	pil_is_kw_icl 				();
	PIL_BOOL 	pil_is_kw_it8_1 			();
	PIL_BOOL 	pil_is_kw_it8_2 			();
	PIL_BOOL 	pil_is_kw_ms_rtf 			();				
	PIL_BOOL 	pil_is_kw_ms_word_3 		();			
	PIL_BOOL 	pil_is_kw_ms_word_4 		();			
	PIL_BOOL 	pil_is_kw_mif 				();					
	PIL_BOOL 	pil_is_kw_pict 				();				
	PIL_BOOL 	pil_is_kw_pil_layout_ntet 	();			
	PIL_BOOL 	pil_is_kw_postscript 		();			
	PIL_BOOL 	pil_is_kw_quark_xpresstags 	();	
	PIL_BOOL 	pil_is_kw_scitex_ct 		();			
	PIL_BOOL 	pil_is_kw_scitex_handshake 	();	
	PIL_BOOL 	pil_is_kw_scitex_lw 		();			
	PIL_BOOL 	pil_is_kw_sgml 				();				
	PIL_BOOL 	pil_is_kw_tiff 				();				
	PIL_BOOL 	pil_is_kw_wp_100 			();				
	PIL_BOOL 	pil_is_kw_wp_102 			();				
	PIL_BOOL 	pil_is_kw_xyquest_xywrite 	();		
	PIL_BOOL 	pil_is_kw_mac_filename 		();		
	PIL_BOOL 	pil_is_kw_unix_filename 	();		
	PIL_BOOL 	pil_is_kw_msdos_filename 	();		
	PIL_BOOL 	pil_is_kw_os2_filename 		();		
	PIL_BOOL 	pil_is_kw_vms_filename 		();		
	PIL_BOOL 	pil_is_kw_vm_filename 		();			
	PIL_BOOL 	pil_is_kw_lcd_filename 		();		
	PIL_BOOL 	pil_is_kw_inline 			();				
	PIL_BOOL 	pil_is_kw_name_table 		();		
	PIL_BOOL	pil_is_kw_nt_type			();			
	PIL_BOOL	pil_is_kw_nt_domain			();
	PIL_BOOL	pil_is_kw_nt_value			();
	PIL_BOOL	pil_is_kw_canvas			();
	PIL_BOOL	pil_is_kw_dimensions		();
	PIL_BOOL	pil_is_kw_units				();
	PIL_BOOL	pil_is_kw_user_name			();
	PIL_BOOL	pil_is_kw_type				();
	PIL_BOOL	pil_is_kw_clipper			();
	PIL_BOOL	pil_is_kw_application		();
	PIL_BOOL	pil_is_kw_app_name			();
	PIL_BOOL	pil_is_kw_app_type			();
	PIL_BOOL	pil_is_kw_app_int8			();
	PIL_BOOL	pil_is_kw_app_int16			();
	PIL_BOOL	pil_is_kw_app_int32			();
	PIL_BOOL	pil_is_kw_app_uint8			();
	PIL_BOOL	pil_is_kw_app_uint16		();
	PIL_BOOL	pil_is_kw_app_uint32		();
	PIL_BOOL	pil_is_kw_app_double		();
	PIL_BOOL	pil_is_kw_app_string		();
	PIL_BOOL	pil_is_kw_fill_rule			();
	PIL_BOOL	pil_is_kw_rectangle			();
	PIL_BOOL	pil_is_kw_circle			();
	PIL_BOOL	pil_is_kw_ellipse			();
	PIL_BOOL	pil_is_kw_polygon			();
	PIL_BOOL	pil_is_kw_closed_path		();
	PIL_BOOL	pil_is_kw_even_odd			();
	PIL_BOOL	pil_is_kw_nz_winding		();
	PIL_BOOL	pil_is_kw_move_to			();
	PIL_BOOL	pil_is_kw_line_to			();
	PIL_BOOL	pil_is_kw_curve_to			();
	PIL_BOOL	pil_is_kw_arc_to			();
	PIL_BOOL	pil_is_kw_flow_label		();
	PIL_BOOL	pil_is_kw_flow_objects		();
	PIL_BOOL	pil_is_kw_flow_from			();
	PIL_BOOL	pil_is_kw_flow_to			();
	PIL_BOOL	pil_is_kw_flow_content		();
	PIL_BOOL	pil_is_kw_id				();
	PIL_BOOL	pil_is_kw_group_objects		();
	PIL_BOOL	pil_is_kw_origin			();
	PIL_BOOL	pil_is_kw_rot_angle			();
	PIL_BOOL	pil_is_kw_rot_point			();
	PIL_BOOL	pil_is_kw_container			();
	PIL_BOOL	pil_is_kw_graphic			();
	PIL_BOOL	pil_is_kw_rc_type			();
	PIL_BOOL	pil_is_kw_rc_name			();
	PIL_BOOL	pil_is_kw_bbox				();
	PIL_BOOL	pil_is_kw_rc_type_text		();
	PIL_BOOL	pil_is_kw_rc_type_geometry	();
	PIL_BOOL	pil_is_kw_rc_type_lineart	();
	PIL_BOOL	pil_is_kw_rc_type_image		();
	PIL_BOOL	pil_is_kw_color_model		();
	PIL_BOOL	pil_is_kw_dev_gray			();
	PIL_BOOL	pil_is_kw_cal_gray			();
	PIL_BOOL	pil_is_kw_dev_rgb			();
	PIL_BOOL	pil_is_kw_cal_rgb			();
	PIL_BOOL	pil_is_kw_dev_cmyk			();
	PIL_BOOL	pil_is_kw_spot				();
	PIL_BOOL	pil_is_kw_pantone			();
	PIL_BOOL	pil_is_kw_render_op			();
	PIL_BOOL	pil_is_kw_stroke			();
	PIL_BOOL	pil_is_kw_fill				();
	PIL_BOOL	pil_is_kw_fill_stroke		();
	PIL_BOOL	pil_is_kw_render_attr		();
	PIL_BOOL	pil_is_kw_stroke_color		();
	PIL_BOOL	pil_is_kw_position			();
	PIL_BOOL	pil_is_kw_width				();
	PIL_BOOL	pil_is_kw_cap				();
	PIL_BOOL	pil_is_kw_join				();
	PIL_BOOL	pil_is_kw_miter_limit		();
	PIL_BOOL	pil_is_kw_fill_color		();
	PIL_BOOL	pil_is_kw_inside			();
	PIL_BOOL	pil_is_kw_outside			();
	PIL_BOOL	pil_is_kw_centered			();
	PIL_BOOL	pil_is_kw_butt_cap			();
	PIL_BOOL	pil_is_kw_round_cap			();
	PIL_BOOL	pil_is_kw_projecting_cap	();
	PIL_BOOL	pil_is_kw_miter_join		();
	PIL_BOOL	pil_is_kw_bevel_join		();
	PIL_BOOL	pil_is_kw_round_join		();
	PIL_BOOL	pil_is_kw_line				();
	PIL_BOOL	pil_is_kw_arc				();
	PIL_BOOL	pil_is_kw_bezier			();
	PIL_BOOL	pil_is_kw_poly_line			();
	PIL_BOOL	pil_is_kw_open_path			();
	PIL_BOOL	pil_is_kw_cm_value			();
	PIL_BOOL	pil_is_kw_rop_value			();
	PIL_BOOL	pil_is_kw_data				();
	PIL_BOOL	pil_is_kw_header			();
	PIL_BOOL	pil_is_kw_little_endian		();
	PIL_BOOL	pil_is_kw_big_endian		();
	PIL_BOOL	pil_is_kw_ascii				();
	PIL_BOOL	pil_is_kw_binary			();
	PIL_BOOL 	pil_is_kw_text				();
	PIL_BOOL 	pil_is_kw_picture			();
	PIL_BOOL 	pil_is_kw_primitive			();
	PIL_BOOL 	pil_is_kw_clipper_shape		();
	PIL_BOOL 	pil_is_kw_container_shape	();
	
	
	/* -------------------------------------------- */
	/* Component and sub-component parser functions */
	/* -------------------------------------------- */
	
	PIL_ERROR pil_parse_nametable 		();
	PIL_ERROR pil_parse_canvas 			();
	PIL_ERROR pil_parse_object 			();
	PIL_ERROR pil_parse_text_flow 		();
	PIL_ERROR pil_parse_group 			();
	PIL_ERROR pil_parse_layout_start	();
	PIL_ERROR pil_parse_dimensions 		();
	PIL_ERROR pil_parse_units			();
	PIL_ERROR pil_parse_user_name 		();
	PIL_ERROR pil_parse_type 			();
	PIL_ERROR pil_parse_asi		 		();
	PIL_ERROR pil_parse_app_attr		();
	PIL_ERROR pil_parse_rectangle		();
	PIL_ERROR pil_parse_circle			();
	PIL_ERROR pil_parse_ellipse			();
	PIL_ERROR pil_parse_move_to			();
	PIL_ERROR pil_parse_line_to			();
	PIL_ERROR pil_parse_point			();
	PIL_ERROR pil_parse_line			();
	PIL_ERROR pil_parse_arc				();
	PIL_ERROR pil_parse_bezier			();
	PIL_ERROR pil_parse_poly_line		();
	PIL_ERROR pil_parse_open_path		();
	PIL_ERROR pil_parse_stroke_position	();
	PIL_ERROR pil_parse_stroke_width	();
	PIL_ERROR pil_parse_stroke_cap		();
	PIL_ERROR pil_parse_stroke_join		();
	PIL_ERROR pil_parse_stroke_mlimit	();
	PIL_ERROR pil_parse_fill_rule		();
	PIL_ERROR pil_parse_flow_label		();
	PIL_ERROR pil_parse_flow_objects	();
	PIL_ERROR pil_parse_flow_from		();
	PIL_ERROR pil_parse_flow_to			();
	PIL_ERROR pil_parse_flow_content	();
	PIL_ERROR pil_parse_group_id		();
	PIL_ERROR pil_parse_group_objects	();
	PIL_ERROR pil_parse_origin 			();
	PIL_ERROR pil_parse_id 				();
	PIL_ERROR pil_parse_rot_angle 		();
	PIL_ERROR pil_parse_rot_point 		();
	PIL_ERROR pil_parse_graphic 		();
	PIL_ERROR pil_parse_rc_type 		();
	PIL_ERROR pil_parse_rc_name 		();
	PIL_ERROR pil_parse_bbox 			();
	PIL_ERROR pil_parse_cnt_start 		();
	PIL_ERROR pil_parse_cnt_header 		();
	PIL_ERROR pil_parse_ascii_cnt 		();
	PIL_ERROR pil_parse_binary_cnt 		();
	PIL_ERROR pil_parse_simp_closed_prim(); 	
	PIL_ERROR pil_parse_polygon			();		
	PIL_ERROR pil_parse_closed_path		();	
	PIL_ERROR pil_parse_curve_to		();		
	PIL_ERROR pil_parse_arc_to			();	
	PIL_ERROR pil_parse_color_value		();
	
	
	/* --------------------------------------- */
	/* Parser functions for structure creation */
	/* --------------------------------------- */
	
	PIL_ERROR pil_make_layout_start_comp 	();
	PIL_ERROR pil_make_layout_end_comp 		();
	PIL_ERROR pil_make_name_table_comp 		();
	PIL_ERROR pil_make_name_table_entry 	();
	PIL_ERROR pil_make_canvas_comp 			();
	PIL_ERROR pil_make_text_flow_comp		();
	PIL_ERROR pil_make_group_comp			();
	PIL_ERROR pil_make_object_comp 			();
	PIL_ERROR pil_make_cnt_start_comp 		();
	PIL_ERROR pil_make_cnt_end_comp 		();
	PIL_ERROR pil_make_cnt_header_comp		();
	
	
	/* ------------------------ */
	/* Parser utility functions */
	/* ------------------------ */
	
	PIL_VOID	pil_copy_to_buf1		();
	PIL_VOID	pil_copy_to_buf2		();
	PIL_VOID	pil_copy_to_buf3		();
	PIL_VOID	pil_copy_to_buf4		();
	PIL_ERROR 	pil_spinner				();
	PIL_BOOL	pil_str_eq				();
	PIL_UINT8	pil_combine_hex_digits	();
	PIL_BOOL	pil_is_pil_file			();
	PIL_BOOL	pil_is_desired_asi		();
	PIL_BOOL	pil_is_hex_digit		();
	PIL_BOOL	pil_is_white_space		();
	PIL_BOOL	pil_is_graphic_prim		();


	/* ------------------------------------------------ */
	/* Generator prototypes								*/
	/* ------------------------------------------------ */

	
	char PTRTYPE *pil_token_to_str			();
	char PTRTYPE *pil_gen_indents			();
	PIL_ERROR 	pil_put_arc					();
	PIL_ERROR 	pil_put_asa					();
	PIL_ERROR 	pil_put_asi					();
	PIL_ERROR 	pil_put_bezier				();
	PIL_ERROR 	pil_put_canvas				();
	PIL_ERROR 	pil_put_circledata			();
	PIL_ERROR 	pil_put_clipper				();
	PIL_ERROR 	pil_put_closed_shape		();
	PIL_ERROR 	pil_put_color				();
	PIL_ERROR 	pil_put_container			();
	PIL_ERROR 	pil_put_content_data		();
	PIL_ERROR 	pil_put_content_end			();
	PIL_ERROR 	pil_put_content_hdr			();
	PIL_ERROR 	pil_put_content_start		();
	PIL_ERROR 	pil_put_ctype_code			();
	PIL_ERROR 	pil_put_data_code			();
	PIL_ERROR 	pil_put_dimensions			();
	PIL_ERROR 	pil_put_domain_code			();
	PIL_ERROR 	pil_put_graphic				();
	PIL_ERROR 	pil_put_group				();
	PIL_ERROR 	pil_put_layout_end			();
	PIL_ERROR 	pil_put_layout_start		();
	PIL_ERROR 	pil_put_line				();
	PIL_ERROR 	pil_put_nametbleentry		();
	PIL_ERROR 	pil_put_name_table			();
	PIL_ERROR 	pil_put_object				();
	PIL_ERROR 	pil_put_objid				();
	PIL_ERROR 	pil_put_pathdata			();
	PIL_ERROR 	pil_put_polydata			();
	PIL_ERROR 	pil_put_pvalue				();
	PIL_ERROR 	pil_put_rectdata			();
	PIL_ERROR 	pil_put_shape				();
	PIL_ERROR 	pil_put_shape_code			();
	PIL_ERROR 	pil_put_text_flow			();
	PIL_VOID	pil_ftoa					();
	PIL_VOID	pil_i32toa					();
	PIL_VOID	pil_nextline				();	
	PIL_VOID	pil_put_dbl					();
	PIL_VOID	pil_put_hex					();
	PIL_VOID	pil_put_i32					();
	PIL_VOID	pil_put_kw					();
	PIL_VOID	pil_put_kw_no_sep			();
	PIL_VOID	pil_put_literal				();
	PIL_VOID	pil_put_named_begblk		();
	PIL_VOID	pil_put_named_dbl			();
	PIL_VOID	pil_put_named_i32			();
	PIL_VOID	pil_put_named_str			();
	PIL_VOID	pil_put_named_ui32			();
	PIL_VOID	pil_put_nested_begblk		();
	PIL_VOID	pil_put_nested_endblk		();
	PIL_VOID	pil_put_point				();
	PIL_VOID	pil_put_point_data			();
	PIL_VOID	pil_put_quoted_str			();
	PIL_VOID	pil_put_string				();
	PIL_VOID	pil_put_ui32				();
	PIL_VOID	pil_to_hex					();
	PIL_VOID	pil_ui32toa					();
	PIL_VOID	pil_set_dflt_hdr			();
#endif
