/*===========================================================================
SOLAR slrreply :: Module ..\slrreply\reply.c
Original Author : Kevin Houle <kjhoule@iowegia.dsm.ia.us>

This source code has been placed into the public domain.
===========================================================================*/

/* Header files */
#include <stdio.h>
#include <process.h>
#include <string.h>
#include <time.h>
#include <dir.h>
#include <stdlib.h>
#include "reply.h"

/*
 * Function: int process_packet()
 * Purpose : Process MSG files as directed by REPLIES.
 * Return  : Number of MSG files processed successfully.
*/

int process_packet()
{
  FILE *reply_file = NULL;
  FILE *msg_file   = NULL;
  int  news_count  = 0;
  int  mail_count  = 0;
  long byte_count  = 0L;
  long msg_size    = 0L;
  char msg_path[MAXPATH];

  /* Initialize global variables */

  strcpy(reply_kind,"NONE");
  strcpy(prefix,"NONE");

  /* Open the REPLIES file for processing */

  if ((reply_file = fopen(REPLY_NAME,"rt")) == NULL)
  {
    RPrintf("No %s file to process\n",REPLY_NAME);
    goto GoodExit;
  }

  /* Process the reply packet MSG files as directed by REPLIES file */

  RPrintf("\nProcessing message files...\n");
  while (read_reply_file(reply_file) == 0)
  {

    /* Build filename for MSG file */

    strcpy(msg_path,prefix);   /* Value for prefix set in read_reply_file */
    strcat(msg_path,".");
    strcat(msg_path,BATCH_EXT);
    RPrintf("\nMsg file : %s : ",msg_path);

    /* Check to make sure MSG file is a supported SOUP format
       reply kind; either news or mail */

    if ((strcmp(reply_kind,"mail") != 0) && \
        (strcmp(reply_kind,"news") != 0))
    {

      /* Unsupported SOUP MSG file, move into lost+found */

      RPrintf("Unknown reply kind=%s : not processed\n",reply_kind);
      if (lost_message(msg_path) == 0)
        RPrintf("-Saved %s to lost+found\n",msg_path);
      else
        RPrintf("-Message %s lost: %s\n",msg_path,_slrerr);
    }
    else   /* The MSG format is supported */
    {

      /* Open the MSG file for processing. If we can't,
         that ain't good. Try to save to lost+found and
         life goes on. */

      RPrintf("processing as %s\n",reply_kind);
      if ((msg_file = fopen(msg_path,"rb")) == NULL)
      {
        RPrintf("slrreply: Message %s is child-proof, cannot open\n",msg_path);
        if (lost_message(msg_path) == 0)
          RPrintf("-Saved %s to lost+found\n",msg_path);
        else
          RPrintf("-Message %s lost: %s\n",msg_path,_slrerr);
        goto NextBatch;
      }

      /* Call the extract function to do it's thing. This
         pulls one or more messages from MSG file, so the
         msg_file must remain open. */

      while ((msg_size = extract_msg(msg_file)) != 0L)
      {
        if (msg_size > 0L) /* This would be a normal result */
        {

          /* Figure out what program to pass the message to
             based on it's SOUP msg type */

          switch (msg_type) {

            /* Binary news message. */

            case 'B'  : if (feed_news() != 0)
                        {
                          if (msg_file) fclose(msg_file);
                          RPrintf("feed_news(): %s\n",_slrerr);
                          if (lost_message(msg_path) == 0)
                            RPrintf("-Saved %s to lost+found\n\n",msg_path);
                          else
                            RPrintf("-Message %s lost: %s\n\n",msg_path,_slrerr);
                          unlink(REPLY_TMP);
                          goto NextBatch;
                        }
                        else
                        {
                          news_count++;
                          byte_count += msg_size;
                          RPrintf("News message posted: %lu bytes\n",msg_size);
                        }
                        break;

            /* Binary mail message, call mail program */

            case 'b'  : if (feed_mail() != 0)
                        {
                          if (msg_file) fclose(msg_file);
                          RPrintf("feed_mail(): %s\n",_slrerr);
                          if (lost_message(msg_path) == 0)
                            RPrintf("-Saved %s to lost+found\n\n",msg_path);
                          else
                            RPrintf("-Message %s lost: %s\n\n",msg_path,_slrerr);
                          unlink(REPLY_TMP);
                          unlink(REPLY_FINAL);
                          goto NextBatch;
                        }
                        else
                        {
                          mail_count++;
                          byte_count += msg_size;
                        }
                        break;

            /* Unsupported message type found */

            default   : RPrintf("Unsupported message type: %c\n",msg_type);
                        if (msg_file) fclose(msg_file);
                        if (lost_message(msg_path) == 0)
                          RPrintf("-Saved %s to lost+found\n\n",msg_path);
                        else
                          RPrintf("-Message %s lost: %s\n\n",msg_path,_slrerr);
                        goto NextBatch;
          }
        }
        else  /* Something abnormal happened during message extraction. */
        {
          if (msg_size == -1L)  /* Non-fatal error flag */
          {
            RPrintf("Skipping msg in %s : %s\n",msg_path,_slrerr);
            sprintf(logbuf,"%s %s",username,_slrerr);
            logit("slrreply");
          }
          else
          {
            /* Fatal error, close file and stop processing it. */

            if (msg_file) fclose(msg_file);
            RPrintf("%s\n",_slrerr);
            sprintf(logbuf,"%s %s",username,_slrerr);
            logit("slrreply");
            if (lost_message(msg_path) != 0)
              RPrintf("-Message lost: %s\n",_slrerr);
            else
              RPrintf("-Saved %s to lost+found\n\n",msg_path);
            goto NextBatch;
          }
        }
      }
      /* Reply packet processed 'normally', close the msg_file */

      if (msg_file) fclose(msg_file);
    }

    /* Delete MSG file and temp files and move on */

NextBatch:
    unlink(msg_path);
    unlink(REPLY_TMP);
    unlink(REPLY_FINAL);
  }

  /* All finished, close up the REPLIES file */

  fclose(reply_file);
  RPrintf("\nPacket processing complete: %u news, %u mail, %lu bytes\n",news_count,mail_count,byte_count);
  sprintf(logbuf,"%s %u news %u mail %lu bytes",username,news_count,mail_count,byte_count);
  logit("slrreply");

GoodExit:
  unlink(REPLY_NAME);
  unlink(REPLY_TMP);
  unlink(REPLY_FINAL);
  return (mail_count + news_count);

ErrorExit:

  /* Something bad happened. Make sure the files are
     closed and try to save the packet to lost+found */

  if (reply_file) fclose(reply_file);
  if (msg_file)   fclose(msg_file);
  fprintf(stdout,"%s\n",_slrerr);
  sprintf(logbuf,"%s",_slrerr);
  logit("slrreply");

  if (lost_packet() != 0)
  {
    RPrintf("Packet lost: %s\n",_slrerr);
    sprintf(logbuf,"%s",_slrerr);
    logit("slrreply");
  }
  else
  {
    RPrintf("Packet saved in lost+found\n");
  }
  return 0;
}

/*
 * Function: read_reply_file()
 * Purpose : Read the REPLIES file and store values.
 * Return  : 0 on success, non-zero on error. Do not set _slrerr,
 *           as error conditions are not trapped from above.
*/

int read_reply_file(FILE *reply_file)
{
  char buf[128];
  char *p;

  /* Read a line from reply file */

  if (fgets(buf,128,reply_file) == NULL)
    goto ErrorExit;

  /* Parse out first value, prefix */

  if ((p = strtok(buf,"\t")) == NULL)
  {
    RPrintf("%s: Unknown file format.\n",REPLY_NAME);
    goto ErrorExit;
  }
  strcpy(prefix,p);

  /* Parse out second value, reply kind */

  if ((p = strtok(NULL,"\t")) == NULL)
  {
    RPrintf("%s: Unknown file format.\n",REPLY_NAME);
    goto ErrorExit;
  }
  strcpy(reply_kind,p);

  /* Parse out third value, message type */

  if ((p = strtok(NULL,"\t")) == NULL)
  {
    RPrintf("%s: Unknown file format.\n",REPLY_NAME);
    goto ErrorExit;
  }
  msg_type = p[0];

GoodExit:
  return 0;
ErrorExit:
  return 1;
}
